/*____________________________________________________________________________
        Copyright (C) 2000 Networks Associates Technology, Inc.
        All rights reserved.

		low level functions for keydb objects
		
        $Id: pgpKeyObj.c,v 1.91.2.3 2001/05/29 17:47:30 hal Exp $
____________________________________________________________________________*/
#include "pgpConfig.h"
#include <string.h>

#include <ctype.h>	/* For tolower() */

#include "pgpDebug.h"
#include "pgpKeyPriv.h"
#include "pgpErrors.h"
#include "pgpPubKey.h"
#include "pgpPktByte.h"
#include "pgpSigSpec.h"
#include "pgpMem.h"
#include "pgpContext.h"
#include "pgpEnv.h"
#include "pgpP11Key.h"
#include "pgpTrustPriv.h"
#include "pgpX509Priv.h"
#include "pgpHashPriv.h"
#include "pgpRegExp.h"
#include "pgpSymmetricCipherPriv.h"
#include "pgpPassCach.h"

#ifndef NULL
#define NULL 0
#endif


/* Return the type of a PGPKeyDBObj */
	PGPUInt32
pgpObjectType(PGPKeyDBObj const *obj)
{
	pgpAssert (obj);
	if (obj->objflags & kPGPKeyDBObjType_Key )
		return RINGTYPE_KEY;
	if (obj->objflags & kPGPKeyDBObjType_SubKey )
		return RINGTYPE_SUBKEY;
	if (obj->objflags & kPGPKeyDBObjType_UserID )
		return RINGTYPE_USERID;
	if (obj->objflags & kPGPKeyDBObjType_Signature )
		return RINGTYPE_SIG;
	if (obj->objflags & kPGPKeyDBObjType_CRL )
		return RINGTYPE_CRL;
	return RINGTYPE_UNK;
}


	static DataHeader **
sObjectDataPointer( PGPKeyDBObj const *obj )
{
	PGPKeyInfo *kinfo;
	PGPUserIDInfo *uinfo;
	PGPSigInfo *sinfo;
	PGPCRLInfo *cinfo;
	PGPUnkInfo *xinfo;
	DataHeader **dptr = NULL;

	switch( pgpObjectType( obj ) )
	{
	case RINGTYPE_KEY:
	case RINGTYPE_SUBKEY:
		kinfo = pgpKeyToKeyInfo( obj );
		dptr = &kinfo->data;
		break;
	case RINGTYPE_USERID:
		uinfo = pgpUserIDToUserIDInfo( obj );
		dptr = &uinfo->data;
		break;
	case RINGTYPE_SIG:
		sinfo = pgpSigToSigInfo( obj );
		dptr = &sinfo->data;
		break;
	case RINGTYPE_CRL:
		cinfo = pgpCRLToCRLInfo( obj );
		dptr = &cinfo->data;
		break;
	case RINGTYPE_UNK:
		xinfo = pgpUnkToUnkInfo( obj );
		dptr = &xinfo->data;
		break;
	default:
		pgpAssert( 0 );
		break;
	}
	return dptr;
}


/* Return pointer to obj's info id field */
	static PGPUInt32 *
sObjectIDPointer( PGPKeyDBObj const *obj )
{
	PGPKeyInfo *kinfo;
	PGPUserIDInfo *uinfo;
	PGPSigInfo *sinfo;
	PGPCRLInfo *cinfo;
	PGPUnkInfo *xinfo;
	PGPUInt32 *idptr = NULL;

	switch( pgpObjectType( obj ) )
	{
	case RINGTYPE_KEY:
	case RINGTYPE_SUBKEY:
		kinfo = pgpKeyToKeyInfo( obj );
		idptr = &kinfo->id;
		break;
	case RINGTYPE_USERID:
		uinfo = pgpUserIDToUserIDInfo( obj );
		idptr = &uinfo->id;
		break;
	case RINGTYPE_SIG:
		sinfo = pgpSigToSigInfo( obj );
		idptr = &sinfo->id;
		break;
	case RINGTYPE_CRL:
		cinfo = pgpCRLToCRLInfo( obj );
		idptr = &cinfo->id;
		break;
	case RINGTYPE_UNK:
		xinfo = pgpUnkToUnkInfo( obj );
		idptr = &xinfo->id;
		break;
	default:
		pgpAssert( 0 );
		break;
	}
	return idptr;
}

/* Return ID corresponding to front end object */
	PGPUInt32
pgpKeyDBObjID( PGPKeyDBObj const *obj )
{
	if( obj->objflags & kPGPKeyDBObjFlags_ID )
		return obj->idinfo.id;
	else
		return *sObjectIDPointer(obj);
}

/* True if key belongs to a front-end keydb */
	PGPBoolean
pgpFrontEndKey( PGPKeyDBObj const *obj )
{
	PGPKeyDB *db = PGPPeekKeyDBObjKeyDB( (PGPKeyDBObj *)obj );
	return pgpFrontEndKeyDB( db );
}


	static DataHeader *
sObjectDataHeader( PGPKeyDBObj const *obj )
{
	return *sObjectDataPointer( obj );
}


/* Refcount the data buffers */
void
pgpObjectHold(PGPKeyDBObj *obj)
{
	DataHeader *objdata;

	objdata = sObjectDataHeader( obj );
	if( IsntNull( objdata ) )
		++objdata->refcount;
}

void
pgpObjectRelease(PGPKeyDBObj *obj)
{
	DataHeader *objdata;

	objdata = sObjectDataHeader( obj );
	if( IsntNull( objdata )  &&  objdata->refcount > 0)
		--objdata->refcount;
}

	PGPByte const *
pgpFetchObject( PGPKeyDBObj const *obj, PGPSize *len )
{
	DataHeader *objdata;

	objdata = sObjectDataHeader( obj );
	if( IsNull( objdata ) )
	{
		PGPByte *buf;
		PGPSize buflen;
		PGPKeyDB *kdb = PGPPeekKeyDBObjKeyDB( (PGPKeyDBObj *)obj );
		PGPContextRef context = PGPPeekKeyDBContext( kdb );
		pgpFetchObjectData_back( context, pgpKeyDBObjID( obj ),
								 &buf, &buflen );
		objdata = pgpNewObjData( kdb, buf, buflen );
		PGPFreeData( buf );
		*sObjectDataPointer( obj ) = objdata;
		
	}
	*len = objdata->len;
	return objdata->data;
}


/* Return a new data header suitable for assigning to ->data */
	void *
pgpNewObjData( PGPKeyDB *kdb, PGPByte const *buf, PGPSize len )
{
	DataHeader *data;
	PGPByte *databuf;
	MemPool cutback;

	cutback = kdb->objPool;
	
	data = memPoolAlloc( &kdb->objPool, sizeof(DataHeader), 4 );
	if( data == NULL )
		goto outofmem;

	databuf = memPoolAlloc( &kdb->objPool, len, 1 );
	if( databuf == NULL )
		goto outofmem;
	pgpCopyMemory( buf, databuf, len );

	data->data = databuf;
	data->len = len;
	data->refcount = 1;
	return data;

 outofmem:
	memPoolCutBack( &kdb->objPool, &cutback );
	pgpKeyDBSetError( kdb, kPGPError_OutOfMemory );
	return NULL;
}

/* Change the data buffer associated with an object */
	PGPError
pgpUpdateObjData( PGPKeyDBObj *obj, PGPByte *buf, PGPSize len )
{
	DataHeader *objdata;
	PGPKeyDB *kdb;
	PGPByte *databuf;

	objdata = sObjectDataHeader( obj );
	if( IsNull( objdata ) )
	{
		return pgpKeyDBError( PGPPeekKeyDBObjKeyDB( obj ) );
	}

	kdb = PGPPeekKeyDBObjKeyDB( obj );
	databuf = memPoolAlloc( &kdb->objPool, len, 1 );
	if( databuf == NULL )
		return kPGPError_OutOfMemory;

	pgpCopyMemory( buf, databuf, len );
	objdata->data = databuf;
	objdata->len = len;

	return kPGPError_NoErr;
}


/*** Access functions for information about objects ***/

	PGPUInt32
pgpKeyBits(PGPKeyDBObj *key)
{
	PGPKeyInfo *kinfo;

	pgpAssert(OBJISKEY(key));
	kinfo = pgpKeyToKeyInfo( key );
	return kinfo->keybits;
}

	PGPUInt32
pgpKeyCreation(PGPKeyDBObj *key)
{
	PGPKeyInfo *kinfo;

	pgpAssert(OBJISKEY(key));
	kinfo = pgpKeyToKeyInfo( key );
	return kinfo->creationtime;
}

	PGPUInt32
pgpKeyExpiration(PGPKeyDBObj *key)
{
	PGPKeyInfo *kinfo;

	pgpAssert(OBJISKEY(key));
	kinfo = pgpKeyToKeyInfo( key );
	if (kinfo->creationtime == 0 || kinfo->validityperiod == 0)
		return 0;    /* valid indefinitely */
	else
		return kinfo->creationtime + (kinfo->validityperiod * 3600 * 24);
}

/*
 * If called for a subkey, force to just encryption.
 * If called for a key with a subkey, return the "or" of both.
 * Else just do the key itself.
 * Internal form of pgpKeyUse - if unExpired is true, check that the
 * subkeys are unexpired before saying it has encryption usage.
 * pgpKeyUse and pgpKeyUnexpiredUse are macros that call this now.
 */
	PGPUInt32
pgpKeyUseInternal(PGPKeyDBObj *key, PGPBoolean unExpired, PGPBoolean revokedOK)
{
	PGPKeyInfo *kinfo;
	int use;
	int first = TRUE;

	pgpAssert(OBJISKEY(key));
	kinfo = pgpKeyToKeyInfo( key );

	use = pgpKeyAlgUse(pgpPkalgByNumber(kinfo->pkalg));
	if (OBJISSUBKEY(key))
		use &= PGP_PKUSE_ENCRYPT;
	for (key=key->down; key; key=key->next) {
		if( !pgpKeyDBObjIsReal(key) )
			continue;
		if (OBJISSUBKEY(key) && pgpSubkeyValid(key, unExpired, revokedOK))
		{
			kinfo = pgpKeyToKeyInfo( key );
			if( first )
				use &= PGP_PKUSE_SIGN;
			first = FALSE;
			use |= pgpKeyAlgUse(pgpPkalgByNumber(kinfo->pkalg));
		}
	}
	return use;
}


	PGPByte
pgpKeyTrust(PGPKeyDBObj *key)
{
	PGPKeyInfo *kinfo;

	pgpAssert(OBJISKEY(key));
	kinfo = pgpKeyToKeyInfo( key );
	return pgpMax(kinfo->trust & kPGPKeyTrust_Mask,
				  kinfo->signedTrust & kPGPKeyTrust_Mask);
}

void
pgpKeySetTrust(PGPKeyDBObj *key, PGPByte trust)
{
	PGPKeyInfo *kinfo;

	pgpAssert(OBJISKEY(key));
	kinfo = pgpKeyToKeyInfo( key );
	pgpAssert(trust==kPGPKeyTrust_Unknown  || trust==kPGPKeyTrust_Never ||
	       trust==kPGPKeyTrust_Marginal || trust==kPGPKeyTrust_Complete);
	pgpAssert (!(kinfo->trust & PGP_KEYTRUSTF_BUCKSTOP));
	if (kinfo->trust & (PGP_KEYTRUSTF_REVOKED | PGP_KEYTRUSTF_EXPIRED))
	    return;
	if ((kinfo->trust & kPGPKeyTrust_Mask) != trust) {
		kinfo->trust = (kinfo->trust & ~kPGPKeyTrust_Mask) + trust;
	}
}


/*
 * Used to set a key as an "axiomatic" key, that is, one for which
 * we hold the private key.  This also involves setting each name on that
 * key as having complete validity.
 */
void
pgpKeySetAxiomatic(PGPKeyDBObj *key)
{
	PGPKeyInfo *kinfo;
	PGPUserIDInfo *uinfo;
	PGPKeyDBObj *name = NULL;

    pgpAssert(OBJISKEY(key));
	kinfo = pgpKeyToKeyInfo( key );
    if( !pgpKeyIsSec (key) ||
			(kinfo->trust & PGP_KEYTRUSTF_REVOKED) )
        return;        /* already axiomatic or can't set */
    kinfo->trust &= ~kPGPKeyTrust_Mask;
    kinfo->trust |= (PGP_KEYTRUSTF_BUCKSTOP | kPGPKeyTrust_Ultimate);
	/* Make sure all names have axiomatic confidence */
	for (name=key->down; name; name=name->next) {
		if( !pgpKeyDBObjIsReal(name) )
			continue;
		if (OBJISUSERID(name)) {
			uinfo = pgpUserIDToUserIDInfo( name );
			uinfo->confidence = PGP_NEWTRUST_INFINITE;
		}
	}
}


/*  Reset an axiomatic key.  Trust is set to undefined. */

void
pgpKeyResetAxiomatic (PGPKeyDBObj *key)
{
	PGPKeyInfo *kinfo;
	PGPUserIDInfo *uinfo;
	PGPKeyDBObj *name = NULL;

    pgpAssert(OBJISKEY(key));
	kinfo = pgpKeyToKeyInfo( key );
    kinfo->trust &= ~PGP_KEYTRUSTF_BUCKSTOP;
    kinfo->trust = (kinfo->trust & ~kPGPKeyTrust_Mask) + 
                           kPGPKeyTrust_Undefined;
	/* Make sure all names have undefined confidence */
	for (name=key->down; name; name=name->next) {
		if( !pgpKeyDBObjIsReal(name) )
			continue;
		if (OBJISUSERID(name)) {
			uinfo = pgpUserIDToUserIDInfo( name );
			uinfo->confidence = PGP_NEWTRUST_UNDEFINED;
		}
	}
}


	PGPBoolean
pgpKeyAxiomatic(PGPKeyDBObj *key)
{
	PGPKeyInfo *kinfo;

	pgpAssert(OBJISKEY(key));
	kinfo = pgpKeyToKeyInfo( key );
	return (kinfo->trust & PGP_KEYTRUSTF_BUCKSTOP) != 0;
}


/* Return TRUE if the key is a subkey */
	PGPBoolean
pgpKeyIsSubkey (PGPKeyDBObj const *key)
{
     return OBJISSUBKEY(key) != 0;
}


	PGPBoolean
pgpKeyDisabled(PGPKeyDBObj *key)
{
	PGPKeyInfo *kinfo;

	pgpAssert(OBJISKEY(key));
	kinfo = pgpKeyToKeyInfo( key );
	return (kinfo->trust & PGP_KEYTRUSTF_DISABLED) != 0;
}

void
pgpKeyDisable(PGPKeyDBObj *key)
{
	PGPKeyInfo *kinfo;

	pgpAssert(OBJISKEY(key));
	kinfo = pgpKeyToKeyInfo( key );
	if (!(kinfo->trust & PGP_KEYTRUSTF_DISABLED)) {
		kinfo->trust |= PGP_KEYTRUSTF_DISABLED;
	}
}

void
pgpKeyEnable(PGPKeyDBObj *key)
{
	PGPKeyInfo *kinfo;

	pgpAssert(OBJISKEY(key));
	kinfo = pgpKeyToKeyInfo( key );
	if (kinfo->trust & PGP_KEYTRUSTF_DISABLED) {
		kinfo->trust &= ~PGP_KEYTRUSTF_DISABLED;
	}
}



	PGPBoolean
pgpKeyRevoked(PGPKeyDBObj *key)
{
	PGPKeyInfo *kinfo;

	pgpAssert(OBJISKEY(key));
	kinfo = pgpKeyToKeyInfo( key );
	return (kinfo->trust & PGP_KEYTRUSTF_REVOKED) != 0;
}


/* Flag for keys whose secret part is on a token.  Only for back end. */
void
pgpKeyOnToken(PGPKeyDBObj *key)
{
	PGPKeyInfo *kinfo;

	pgpAssert(OBJISKEY(key));
	pgpAssert(!pgpFrontEndKey(key));
	kinfo = pgpKeyToKeyInfo( key );
	KEYSETTOKEN( kinfo );
	pgpKeySetAxiomatic( key );
}

void
pgpKeyOffToken(PGPKeyDBObj *key)
{
	PGPKeyInfo *kinfo;

	pgpAssert(OBJISKEY(key));
	pgpAssert(!pgpFrontEndKey(key));
	kinfo = pgpKeyToKeyInfo( key );
	KEYCLEARTOKEN( kinfo );
	/* Mark as no longer axiomatic unless we have a secret part */
	if( !KEYISSEC( kinfo ) )
		pgpKeyResetAxiomatic( key );
}

	PGPBoolean
pgpKeyIsOnToken(PGPKeyDBObj *key)
{
	PGPKeyInfo *kinfo;

	pgpAssert(OBJISKEY(key));
	kinfo = pgpKeyToKeyInfo( key );
	return KEYISTOKEN( kinfo );
}

/* Flag for keys whose secret part has been mathematically checked */
void
pgpSecSetValidated(PGPKeyDBObj *sec)
{
	PGPKeyInfo *kinfo;

	pgpAssert(OBJISKEY(sec));
	pgpAssert(pgpKeyIsSec(sec));
	kinfo = pgpKeyToKeyInfo( sec );
	KEYSETVALIDATED( kinfo );
}

void
pgpSecClearValidated(PGPKeyDBObj *sec)
{
	PGPKeyInfo *kinfo;

	pgpAssert(OBJISKEY(sec));
	pgpAssert(pgpKeyIsSec(sec));
	kinfo = pgpKeyToKeyInfo( sec );
	KEYCLEARVALIDATED( kinfo );
}

	PGPBoolean
pgpSecIsValidated(PGPKeyDBObj *sec)
{
	PGPKeyInfo *kinfo;

	pgpAssert(OBJISKEY(sec));
	pgpAssert(pgpKeyIsSec(sec));
	kinfo = pgpKeyToKeyInfo( sec );
	return KEYISVALIDATED( kinfo );
}

/* 
 * Return true if the specified signature has been revoked, that is,
 * if there is a newer signature by the same key which is of type
 * UID_REVOKE, or if it has been revoked by CRL.
 */
	PGPBoolean
pgpSigRevoked (PGPKeyDBObj *sig)
{
	PGPKeyDBObj		*parent,
					*sibling;
	PGPSigInfo *sinfo;
	PGPSigInfo *sibinfo;

	pgpAssert (OBJISSIG(sig));
	sinfo = pgpSigToSigInfo( sig );

	/* Sigs can be declared irrevocable at creation time */
	if (!SIGISREVOCABLE(sinfo))
		return FALSE;

	/* Check revoked flag */
	if (sinfo->trust & PGP_SIGTRUSTF_REVOKEDBYCRL)
		return TRUE;

	parent = sig->up;
	for (sibling = parent->down; sibling ; sibling = sibling->next) {
		if( !pgpKeyDBObjIsReal(sibling) )
			continue;
		if (!OBJISSIG(sibling) || sibling == sig)
			continue;
		sibinfo = pgpSigToSigInfo( sibling );
		if (sibinfo->by == sinfo->by) {
			if (sibinfo->trust & PGP_SIGTRUSTF_CHECKED) {
				if (sibinfo->tstamp > sinfo->tstamp) {
					if (sibinfo->type == PGP_SIGTYPE_KEY_UID_REVOKE) {
						/* Valid revocation */
						return TRUE;
					}
				}
			}
		}
	}
	return FALSE;
}


	PGPUInt16
pgpKeyConfidence(PGPKeyDBObj *key)
{
	pgpAssert(OBJISKEY(key));

	/*  pgpKeyCalcTrust handles revoked/expired keys */
	return pgpKeyCalcTrust (key);
}

void
pgpKeyID8(
	PGPKeyDBObj const *key,
	PGPByte *pkalg,
	PGPKeyID *keyID)
{
	PGPKeyInfo *kinfo;
	PGPByte keypkalg;

	pgpAssert(OBJISKEY(key));
	kinfo = pgpKeyToKeyInfo( key );
	keypkalg = kinfo->pkalg;
	if ((keypkalg | 1) == 3)		/* ViaCrypt */
		keypkalg = 1;
	if (pkalg)
		*pkalg = keypkalg;
	if (keyID)
	{
		pgpNewKeyIDFromRawData( kinfo->keyID, keypkalg, 8, keyID );
	}
}

void
pgpKeyID4(
	PGPKeyDBObj const *key,
	PGPByte *pkalg,
	PGPKeyID *keyID)
{
	PGPKeyInfo *kinfo;
	PGPByte keypkalg;

	pgpAssert(OBJISKEY(key));
	kinfo = pgpKeyToKeyInfo( key );
	keypkalg = kinfo->pkalg;
	if ((keypkalg | 1) == 3)		/* ViaCrypt */
		keypkalg = 1;
	if (pkalg)
		*pkalg = keypkalg;
	if (keyID)
	{
		pgpNewKeyIDFromRawData( &kinfo->keyID[4], keypkalg, 4, keyID );
	}
}

	PGPBoolean
pgpKeyV3(PGPKeyDBObj const *key)
{
	PGPKeyInfo *kinfo;
	pgpAssert(OBJISKEY(key));
	kinfo = pgpKeyToKeyInfo( key );
	return KEYISV3(kinfo) != 0;
}

	PGPError
pgpKeyFingerprint16(PGPKeyDBObj *key,
	PGPByte *buf)
{
	PGPByte const *p;
	PGPContextRef context;
	PGPSize len;

	pgpAssert(OBJISKEY(key));
	context = PGPPeekKeyDBContext(PGPPeekKeyDBObjKeyDB( key ) );

	p = (PGPByte const *)pgpFetchObject(key, &len);
	if (!p)
		return kPGPError_OutOfMemory;
	return pgpKeyParseFingerprint16(context, p, len, buf);
}

	PGPError
pgpKeyFingerprint20(PGPKeyDBObj *key,
	PGPByte *buf)
{
	PGPContextRef context;
	PGPSize objlen;
	PGPSize hlen;
	PGPByte const *objbuf;
	PGPBoolean mustFree = FALSE;
	PGPError err;

	pgpAssert(OBJISKEY(key));
	context = PGPPeekKeyDBContext(PGPPeekKeyDBObjKeyDB( key ) );

	if( pgpKeyIsSec( key ) )
	{
		objbuf = pgpKeyDBObjToPubData( key, &objlen );
		mustFree = TRUE;
	} else {
		objbuf = (PGPByte const *)pgpFetchObject(key, &objlen);
	}
	hlen = pgpPktBufferHeaderLen( objbuf );
	err = pgpFingerprint20HashBuf(context, objbuf+hlen, objlen-hlen, buf);
	if( mustFree )
		PGPFreeData( (PGPByte *)objbuf );
	return err;
}

/* This does a 20 byte fingerprint based solely on the numeric material */
	PGPError
pgpKeyFingerprint20n(PGPKeyDBObj *key,
	PGPByte *buf)
{
	PGPContextRef context;
	PGPSize objlen;
	PGPSize numlen;
	PGPByte const *objbuf, *numbuf;
	PGPBoolean mustfree = FALSE;
	PGPError err;

	pgpAssert(OBJISKEY(key));
	context = PGPPeekKeyDBContext(PGPPeekKeyDBObjKeyDB( key ) );

	if( pgpKeyIsSec (key) ) {
		objbuf = pgpKeyDBObjToPubData( key, &objlen );
		mustfree = TRUE;
	} else {
		objbuf = (PGPByte const *)pgpFetchObject(key, &objlen);
	}
	numbuf = pgpKeyParseNumericData(objbuf, objlen, &numlen);
	err = pgpFingerprint20HashBuf(context, numbuf, numlen, buf);
	if( mustfree )
		PGPFreeData( (PGPByte *)objbuf );
	return err;
}

/* Return TRUE if the key is a secret key */
	PGPBoolean
pgpKeyIsSec(PGPKeyDBObj *key)
{
	PGPKeyInfo *kinfo;

	pgpAssert(OBJISKEY(key));
	kinfo = pgpKeyToKeyInfo( key );
	return KEYISSEC(kinfo)  ||  KEYISTOKEN(kinfo);
}


/*
 * Return the most recent subkey associated with the key, if there is one.
 * The subkey must be in the specified keyset.
 */
	PGPKeyDBObj *
pgpKeySubkey(PGPKeyDBObj const *key, PGPKeySet *set)
{
	PGPKeyDBObj *obj, *best = NULL;
	PGPUInt32 objtime, besttime = 0;

	pgpAssert(OBJISKEY(key));
	for (obj = key->down; IsntNull(obj); obj = obj->next) {
		if( !pgpKeyDBObjIsReal(obj) )
			continue;
		if( !OBJISSUBKEY(obj) )
			continue;
		if( IsntNull(set) && !pgpKeySetIsMember(obj, set ) )
			continue;
		if (pgpSubkeyValid(obj, TRUE, FALSE)) {
			objtime = pgpKeyCreation(obj);
			if (besttime <= objtime) {
				best = obj;
				besttime = objtime;
			}
		}
	}
	return best;
}

/* Given a subkey, return its master key */
	PGPKeyDBObj *
pgpKeyMasterkey (PGPKeyDBObj const *subkey)
{
	pgpAssert (OBJISSUBKEY(subkey));
	pgpAssert (OBJISTOPKEY(subkey->up));
	return subkey->up;
}

/* Return MPI data of key */
	PGPByte const *
pgpKeyMPIData( PGPKeyDBObj const *key, PGPSize *plen )
{
	PGPSize len;
	PGPSize hlen;
	PGPSize offset;
	PGPByte const *buf;

	pgpAssert (OBJISKEY(key));
	pgpAssert( IsntNull( plen ) );
	*plen = 0;

	buf = pgpFetchObject( key, &len );
	if( IsNull( buf ) )
		return buf;
	hlen = pgpPktBufferHeaderLen( buf );
	buf += hlen;
	len -= hlen;
	/* V4 keys are two bytes shorter than V3 */
	offset = 6;
	if (buf[0] <= PGPVERSION_3)
		offset += 2;
	len -= offset;

	*plen = len;
	return buf + offset;
}
	

/*
 * Given a public key on the keyring, get the corresponding PGPPubKey.
 * Use is a usage code which limits the kinds of keys we will accept.
 * For keys which have subkeys this chooses which one to use.  If use is
 * 0 we do a straight conversion of the key or subkey; if nonzero we
 * verify that the key has the required use.  Return NULL if we can't
 * get a key with the required use.
 */
	PGPPubKey *
pgpKeyPubKey(PGPKeyDBObj *key, int use)
{
	PGPKeyInfo *kinfo;
	PGPByte const *p;
	PGPSize len;
	PGPSize hlen;
	PGPPubKey *pub;
	PGPKeyDBObj *subkey = NULL;
	unsigned vsize;
	PGPError i;
	PGPContextRef	context;

	context = PGPPeekKeyDBContext(PGPPeekKeyDBObjKeyDB( key ) );

	pgpAssert(OBJISKEY(key));
	kinfo = pgpKeyToKeyInfo( key );

	/* Select between subkey and key if necessary */
	if (use &&  (OBJISSUBKEY(key)
			|| ((subkey=pgpKeySubkey(key, NULL)) != NULL))) {
		if (use == PGP_PKUSE_SIGN_ENCRYPT) {
			pgpKeyDBSetError(PGPPeekKeyDBObjKeyDB(key),
							 kPGPError_PublicKeyUnimplemented);
			return NULL;
		}
		if (use == PGP_PKUSE_ENCRYPT) {
			/* Always encrypt to subkey if there is one */
			/* Eventually need to make this skip signature subkeys */
			if (OBJISTOPKEY(key)) {
				pgpAssert(subkey);
				key = subkey;
				pgpAssert (OBJISSUBKEY(key));
				kinfo = pgpKeyToKeyInfo( key );
			}
		} else if (use == PGP_PKUSE_SIGN) {
			/* Allow verification with subkey if explicit, else go up */
			if (use && ((pgpKeyAlgUse(pgpPkalgByNumber(kinfo->pkalg)) & use) !=
							(unsigned)use)){
				if (OBJISSUBKEY(key)) {
					key = key->up;
					pgpAssert (OBJISTOPKEY(key));
					kinfo = pgpKeyToKeyInfo( key );
				}
			}
		}
	}

	/* Verify key satisfies required usage */
	if (use && ((pgpKeyAlgUse(pgpPkalgByNumber(kinfo->pkalg)) & use) !=
			 		(unsigned)use)){
		pgpKeyDBSetError(PGPPeekKeyDBObjKeyDB(key),
						 kPGPError_PublicKeyUnimplemented);
		return NULL;
	}

	p = (PGPByte const *)pgpFetchObject(key, &len);
	if (!p)
		return NULL;
	hlen = pgpPktBufferHeaderLen( p );
	p += hlen;
	len -= hlen;

	/*
	 * A key starts with 5 or 7 bytes of data, an algorithm byte, and
	 * the public components.
	 */
	if (p[0] == PGPVERSION_4) {
		vsize = 0;
	} else {
		vsize = 2;
	}
	pgpAssert(p[5+vsize] == kinfo->pkalg);	/* Checked by pgpKeyVerify */
	pub = pgpPubKeyFromBuf( context, p[5+vsize], p+6+vsize, len-6-vsize, &i);
	if (!pub) {
		pgpKeyDBSetError(PGPPeekKeyDBObjKeyDB(key), i);
		return NULL;
	}
	memcpy(pub->keyID, kinfo->keyID, sizeof(kinfo->keyID));
	pub->keyDBObj = key;
	return pub;
}

/*
 * Return the version for a key.
 */
PgpVersion
pgpKeyVersion (PGPKeyDBObj *key)
{
	PGPByte *kdata;
	PGPSize kdatalen;
	PGPSize hlen;

	pgpAssert(OBJISKEY( key ) );

	kdata = (PGPByte *)pgpFetchObject (key, &kdatalen);
	pgpAssert( IsntNull( kdata ) );
	hlen = pgpPktBufferHeaderLen( kdata );
	kdata += hlen;
	kdatalen -= hlen;
	return (PgpVersion)kdata[0];
}


/*
 * Given a secret on a keyring, get a PGPSecKey (possibly locked).
 * As a hack to help the lazy programmer, you can also pass a key.
 * Use is a usage code which limits the kinds of keys we will accept.
 * For keys which have subkeys this chooses which one to use.
 */
	PGPSecKey *
pgpSecSecKey(PGPKeyDBObj *sec, int use)
{
	PGPKeyInfo *kinfo;
	PGPByte const *p;
	PGPSize len;
	PGPSize hlen;
	PGPSecKey *seckey;
	PGPKeyDBObj *subkey = NULL;
	unsigned vsize;
	PGPError i;
	PGPContextRef	context;
	PGPToken *tok = NULL;

	context = PGPPeekKeyDBContext(PGPPeekKeyDBObjKeyDB( sec ) );

	pgpAssert(OBJISKEY(sec));
	pgpAssert(pgpKeyIsSec(sec));
	kinfo = pgpKeyToKeyInfo( sec );

	/* Select between subkey and sec if necessary */
	if (use	 &&  (OBJISSUBKEY(sec)
			|| ((subkey=pgpKeySubkey(sec, NULL)) != NULL))) {
		if (use == PGP_PKUSE_SIGN_ENCRYPT) {
			pgpKeyDBSetError(PGPPeekKeyDBObjKeyDB(sec),
							 kPGPError_PublicKeyUnimplemented);
			return NULL;
		}
		/* For decrypt/sign, we will use the specified key if legal */
		/* This will allow us to decrypt with top level RSA key */
		/* Could also sign with subkeys eventually */
		if (use && ((pgpKeyAlgUse(pgpPkalgByNumber(kinfo->pkalg)) & use) !=
								(unsigned)use)){
			if (use == PGP_PKUSE_ENCRYPT) {
				if (OBJISTOPKEY(sec)) {
					pgpAssert(subkey);
					sec = subkey;
					pgpAssert (OBJISSUBKEY(sec));
					kinfo = pgpKeyToKeyInfo( sec );
				}
			} else if (use == PGP_PKUSE_SIGN) {
				if (OBJISSUBKEY(sec)) {
					sec = sec->up;
					pgpAssert (OBJISTOPKEY(sec));
					kinfo = pgpKeyToKeyInfo( sec );
				}
			}
		}
	}


	/* Verify sec satisfies required usage */
	if (use && ((pgpKeyAlgUse(pgpPkalgByNumber(kinfo->pkalg)) & use) !=
									(unsigned)use)){
		pgpKeyDBSetError(PGPPeekKeyDBObjKeyDB(sec),
						 kPGPError_PublicKeyUnimplemented);
		return NULL;
	}


	/* See if key is on a hardware token */
	if( !KEYISSEC(kinfo) && KEYISTOKEN(kinfo) )
		tok = pgpTokenFromKeyID( kinfo->keyID );

	p = (PGPByte const *)pgpFetchObject(sec, &len);
	if (!p)
		return NULL;
	hlen = pgpPktBufferHeaderLen( p );
	p += hlen;
	len -= hlen;

	if (p[0] == PGPVERSION_4) {
		vsize = 0;
	} else {
		vsize = 2;
	}
	pgpAssert(p[5+vsize] == kinfo->pkalg);
	seckey = pgpSecKeyFromBuf( context, p[5+vsize], p+6+vsize, len-6-vsize,
							   (PGPBoolean)!!KEYISV3(kinfo), tok, &i);
	if (!seckey) {
		pgpKeyDBSetError(PGPPeekKeyDBObjKeyDB(sec), i);
		return NULL;
	}
	memcpy(seckey->keyID, kinfo->keyID, sizeof(kinfo->keyID));
	seckey->keyDBObj = sec;
	return seckey;
}


/* Return the latest valid sig on the object by the specified key */
	PGPKeyDBObj *
pgpLatestSigByKey (PGPKeyDBObj const *obj, PGPKeyDBObj const *key)
{
	PGPKeyDBObj *sig;
	PGPKeyDBObj *latestsig;

	pgpAssert (!OBJISSIG(obj));

	latestsig = NULL;
	for (sig=obj->down; IsntNull(sig); sig=sig->next) {
		if( !pgpKeyDBObjIsReal(sig) )
			continue;
		if (!OBJISSIG(sig))
			continue;
		if (pgpSigMaker (sig) != key)
			continue;
		if ((pgpSigType (sig) & 0xf0) != PGP_SIGTYPE_KEY_GENERIC)
			continue;
		if (!pgpSigChecked (sig) || pgpSigRevoked (sig))
			continue;
		/* Save the newest such signature on the name */
		if (!latestsig) {
			latestsig = sig;
		} else {
			if (pgpSigTimestamp(sig) > pgpSigTimestamp(latestsig))
				latestsig = sig;
		}
	}		
	return latestsig;
}

/* True if there is a newer sig of the same type by the same key */
	PGPBoolean
pgpSigIsSuperceded( PGPKeyDBObj const *sig )
{
	PGPKeyDBObj *obj;
	PGPKeyDBObj *signer;
	PGPUInt32 type;
	PGPSigInfo *sinfo;

	pgpAssert( OBJISSIG(sig) );

	sinfo = pgpSigToSigInfo( sig );
	if( !SIGISREVOCABLE(sinfo) )
		return FALSE;

	signer = pgpSigMaker( sig );
	type = pgpSigType( sig ) & 0xf0;

	if( IsNull( signer ) )
		return FALSE;

	for( obj=sig->up->down; IsntNull(obj); obj=obj->next ) {
		if( obj == sig )
			continue;
		if( !pgpKeyDBObjIsReal(obj) )
			continue;
		if( !OBJISSIG(obj) )
			continue;
		if( pgpSigMaker (obj) != signer )
			continue;
		if( (pgpSigType (obj) & 0xf0) != type )
			continue;
		if( !pgpSigChecked (obj) )
			continue;
		if (pgpSigTimestamp(sig) >= pgpSigTimestamp(obj))
			continue;
		return TRUE;
	}
	return FALSE;
}


/* There ain't much to know about a name... */
char const *
pgpUserIDName(PGPKeyDBObj *name, PGPSize *lenp)
{
	char const *buf;
	PGPSize len;
	PGPSize hlen;

	pgpAssert(OBJISUSERID(name));
	buf = (char const *)pgpFetchObject(name, &len);
	hlen = pgpPktBufferHeaderLen( (PGPByte *) buf );
	buf += hlen;
	len -= hlen;
	if( IsntNull(lenp) )
		*lenp = len;
	return buf;
}

	PGPBoolean
pgpUserIDIsAttribute(PGPKeyDBObj *name)
{
	PGPUserIDInfo *uinfo;

	pgpAssert(OBJISUSERID(name));
	uinfo = pgpUserIDToUserIDInfo( name );

	return (PGPBoolean)NAMEISATTR(uinfo);
}

	PGPUInt32
pgpUserIDCountAttributes(PGPKeyDBObj *name)
{
	PGPUserIDInfo *uinfo;
	PGPByte			*p;
	PGPSize			len;
	
	pgpAssert(OBJISUSERID(name));
	uinfo = pgpUserIDToUserIDInfo( name );
	if (!NAMEISATTR(uinfo))
		return 0;

	p = (PGPByte *)pgpFetchObject(name, &len);
	if (!p) {
		return 0;
	}

	return pgpAttrCountSubpackets (p, len);
}


/* Return the nth attribute subpacket for the specified name */
	PGPByte const *
pgpUserIDAttributeSubpacket (PGPKeyDBObj *name, PGPUInt32 nth,
	PGPUInt32 *subpacktype, PGPSize *plen, PGPError *error)
{
	PGPUserIDInfo *uinfo;
	PGPByte			*p;
	PGPSize			len;
	
	pgpAssert(OBJISUSERID(name));
	uinfo = pgpUserIDToUserIDInfo( name );

	if (error)
		*error = kPGPError_NoErr;

	if (!NAMEISATTR(uinfo)) {
		if (error)
			*error = kPGPError_BadParams;
		return NULL;
	}

	p = (PGPByte *)pgpFetchObject(name, &len);

	return pgpAttrSubpacket(p, len, nth, subpacktype, plen);
}	

/* Return true if userid is self-signed.  If checked is true, it must
 * be checked, otherwise we will accept it as long as it has not failed
 * the signature check.
 */
	PGPBoolean
pgpUserIDIsSelfSigned( PGPKeyDBObj *name, PGPBoolean checked )
{
	PGPKeyDBObj		*signer;
	PGPKeyDBObj		*sig;

	pgpAssert(OBJISUSERID(name));

	signer = name->up;
	pgpAssert( OBJISTOPKEY( signer ) );
	for( sig = name->down; sig; sig = sig->next )
	{
		if( !pgpKeyDBObjIsReal(sig) )
			continue;
		if( !OBJISSIG(sig) )
			continue;
		if( pgpSigMaker (sig) != signer )
			continue;
		if( (pgpSigType (sig) & 0xf0) != PGP_SIGTYPE_KEY_GENERIC )
			continue;
		if( (checked || pgpSigTried(sig)) && !pgpSigChecked(sig) )
			continue;
		if( pgpSigRevoked (sig) )
			continue;
		return TRUE;
	}
	return FALSE;
}


/* Return the primary userid of the specified attribute type */
/* This reads from disk and munges the pktbuf for nonzero attribute */
	PGPKeyDBObj *
pgpKeyPrimaryUserID (PGPKeyDBObj *key, PGPUInt32 type)
{
	PGPKeyDBObj *name;
	PGPKeyDBObj *firstname;
	PGPKeyDBObj *sig;
	PGPKeyDBObj *newestsig;
	PGPUserIDInfo *uinfo;
	PGPSigInfo *sinfo;
	PGPSigInfo *newestsinfo = NULL;
	PGPKeyDBObj *settrustname;
	PGPUInt32 subpacktype;

    pgpAssert(OBJISKEY(key));

	newestsig = NULL;
	firstname = NULL;
	settrustname = NULL;
	for (name=key->down; IsntNull(name); name=name->next) {
		if( !pgpKeyDBObjIsReal(name) )
			continue;
		if (!OBJISUSERID(name))
			continue;
		uinfo = pgpUserIDToUserIDInfo( name );
		if ((type == 0) != !NAMEISATTR(uinfo))
			continue;
		subpacktype = 0;
		if (NAMEISATTR(uinfo)) {
			(void)pgpUserIDAttributeSubpacket (name, 0, &subpacktype,
											  NULL, NULL);
			if (subpacktype != type)
				continue;
		}
		/* Have a name which is the right attribute type */
		if (!firstname)
			firstname = name;
		/* Names set manually by user win over subpacket userids */
		if( uinfo->oldvalidity & PGP_USERIDTRUSTF_PRIMARY )
		{
			settrustname = name;
			break;
		}
		sig = pgpLatestSigByKey (name, key);
		if (sig) {
			pgpAssert (OBJISSIG(sig));
			sinfo = pgpSigToSigInfo( sig );
			if (SIGISPRIMARYUID (sinfo)) {
				if (!newestsig) {
					newestsig = sig;
					newestsinfo = sinfo;
				} else {
					/* Don't override irrevocable settings */
					if (SIGISREVOCABLE(newestsinfo) &&
								(pgpSigTimestamp(sig) >
								 pgpSigTimestamp(newestsig))) {
						newestsig = sig;
						newestsinfo = sinfo;
					}
				}
			}
		}
	}

	if (firstname == NULL)
		return NULL;

	if( settrustname ) {
		name = settrustname;
	} else if (newestsig) {
		name = newestsig->up;
	} else {
		name = firstname;
	}

	pgpAssert (name);
	pgpAssert (OBJISUSERID(name));
	return name;
}


/* Set the specified userid as primary userid in trust field. */
	PGPError
pgpKeySetPrimaryUserID( PGPKeyDBObj *userid )
{
	PGPUserIDInfo *uinfo;
	PGPUInt32 type;

    pgpAssert(OBJISUSERID(userid));
	
	uinfo = pgpUserIDToUserIDInfo( userid );
	if( NAMEISATTR(uinfo) )
	{
		pgpUserIDAttributeSubpacket (userid, 0, &type, NULL, NULL);
	} else {
		type = 0;
	}
	pgpKeyClearPrimaryUserIDs( userid->up, type );
	uinfo->oldvalidity |= PGP_USERIDTRUSTF_PRIMARY;
	return kPGPError_NoErr;
}

/* Clear the primary userid flags from trust bytes of all userids of
 * specified type.  If type == -1UL then do it from all userids.
 */
	PGPError
pgpKeyClearPrimaryUserIDs( PGPKeyDBObj *key, PGPUInt32 type )
{
	PGPKeyDBObj *name;
	PGPUserIDInfo *uinfo;
	PGPUInt32 subpacktype;

    pgpAssert(OBJISTOPKEY(key));

	for (name=key->down; IsntNull(name); name=name->next) {
		if( !pgpKeyDBObjIsReal(name) )
			continue;
		if (!OBJISUSERID(name))
			continue;
		uinfo = pgpUserIDToUserIDInfo( name );
		/* Make sure it is right type */
		if( type != -1UL )
		{
			if ((type == 0) != !NAMEISATTR(uinfo))
				continue;
			subpacktype = 0;
			if (NAMEISATTR(uinfo)) {
				(void)pgpUserIDAttributeSubpacket (name, 0, &subpacktype,
												  NULL, NULL);
				if (subpacktype != type)
					continue;
			}
		}
		uinfo->oldvalidity &= ~PGP_USERIDTRUSTF_PRIMARY;
	}
	return kPGPError_NoErr;
}



/*  Return the old-style 4-level validity of a name */

	PGPByte
pgpUserIDOldValidity(PGPKeyDBObj *name)
{
	PGPKeyInfo *kinfo;
    PGPKeyDBObj *key;
	PGPUserIDInfo *uinfo;
	
    pgpAssert(OBJISUSERID(name));

    key = name->up;
    pgpAssert(OBJISTOPKEY(key));
	kinfo = pgpKeyToKeyInfo( key );
    /*
	 * Force returned value if key is revoked or axiomatic.
	 * Allow expired keys to stay valid, so users can know what their status
	 * was before they expired.
	 */
    if (kinfo->trust & PGP_KEYTRUSTF_REVOKED)
        return kPGPUserIDTrust_Untrusted;
    if (kinfo->trust & PGP_KEYTRUSTF_BUCKSTOP)
        return kPGPUserIDTrust_Complete;
	uinfo = pgpUserIDToUserIDInfo( name );
    return uinfo->oldvalidity & kPGPUserIDTrust_Mask;
}

	PGPBoolean
pgpUserIDWarnonly(PGPKeyDBObj *name)
{
	PGPUserIDInfo *uinfo;

	pgpAssert(OBJISUSERID(name));

	uinfo = pgpUserIDToUserIDInfo( name );
	return uinfo->oldvalidity & PGP_USERIDTRUSTF_WARNONLY;
}

void
pgpUserIDSetWarnonly(PGPKeyDBObj *name)
{
	PGPUserIDInfo *uinfo;

	pgpAssert(OBJISUSERID(name));

	uinfo = pgpUserIDToUserIDInfo( name );
	if (!(uinfo->oldvalidity & PGP_USERIDTRUSTF_WARNONLY)) {
		uinfo->oldvalidity |= PGP_USERIDTRUSTF_WARNONLY;
	}
}

	PGPUInt16
pgpUserIDValidity(PGPKeyDBObj *name)
{
	PGPKeyInfo *kinfo;
    PGPKeyDBObj *key;
	PGPUserIDInfo *uinfo;

    pgpAssert(OBJISUSERID(name));

    key = name->up;
    pgpAssert (OBJISTOPKEY(key));
	kinfo = pgpKeyToKeyInfo( key );
	/*
	 * Force returned value if key is revoked or axiomatic.
	 * Allow expired keys to stay valid, so users can know what their status
	 * was before they expired.
	 */
    if (kinfo->trust & PGP_KEYTRUSTF_REVOKED)
        return 0;
    if (kinfo->trust & PGP_KEYTRUSTF_BUCKSTOP)
        return PGP_TRUST_INFINITE;
	uinfo = pgpUserIDToUserIDInfo( name );
    return pgpTrustToIntern (uinfo->validity); 
}

	PGPUInt16
pgpUserIDConfidence(PGPKeyDBObj *name)
{
	PGPUserIDInfo *uinfo;

	pgpAssert(OBJISUSERID(name));

	uinfo = pgpUserIDToUserIDInfo( name );
	return pgpTrustToIntern (uinfo->confidence);
}

	PGPBoolean
pgpUserIDConfidenceUndefined(PGPKeyDBObj *name)
{
	PGPUserIDInfo *uinfo;

	pgpAssert(OBJISUSERID(name));

	uinfo = pgpUserIDToUserIDInfo( name );
	return (uinfo->confidence == PGP_NEWTRUST_UNDEFINED);
}


void
pgpUserIDSetConfidence(PGPKeyDBObj *name,
		      PGPUInt16 confidence)
{
	PGPUserIDInfo *uinfo;

	pgpAssert(OBJISUSERID(name));

	confidence = (PGPUInt16) pgpTrustToExtern (confidence);
	
	uinfo = pgpUserIDToUserIDInfo( name );
	if (uinfo->confidence != confidence) {
		uinfo->confidence = (PGPByte) confidence;
	}
}


	PGPKeyDBObj *
pgpSigMaker(PGPKeyDBObj const *sig)
{
	PGPKeyDBObj *key;
	PGPSigInfo *sinfo;

	pgpAssert(OBJISSIG(sig));

	sinfo = pgpSigToSigInfo( sig );
	key = sinfo->by;
	if( !pgpKeyDBObjIsReal( key )  ||  ! OBJISTOPKEY( key ) )
		return NULL;
	pgpAssert(OBJISKEY(key));
	return key;
}

/*
 * This is like pgpSigMaker but will return the signer even if dummy or
 * not a top level key.
 */
	PGPKeyDBObj *
pgpSigMakerDummyOK(PGPKeyDBObj const *sig)
{
	PGPSigInfo *sinfo;

	pgpAssert(OBJISSIG(sig));

	sinfo = pgpSigToSigInfo( sig );
	return sinfo->by;
}

void
pgpSigID8(PGPKeyDBObj const *sig, PGPByte *pkalg, PGPKeyID *keyID)
{
	PGPKeyDBObj *key;
	PGPSigInfo *sinfo;
	PGPKeyInfo *kinfo;
	PGPByte keypkalg;

	pgpAssert(OBJISSIG(sig));

	sinfo = pgpSigToSigInfo( sig );
	key = sinfo->by;
	pgpAssert(OBJISKEY(key));
	kinfo = pgpKeyToKeyInfo( key );
	keypkalg = kinfo->pkalg;
	if ((keypkalg | 1) == 3)
		keypkalg = 1;	/* ViaCrypt */
	if (pkalg)
		*pkalg = keypkalg;
	if ( keyID )
	{
		pgpNewKeyIDFromRawData( kinfo->keyID, keypkalg, 8, keyID );
	}
}

	PGPByte
pgpSigTrust(PGPKeyDBObj *sig)
{
	PGPSigInfo *sinfo;

	pgpAssert(OBJISSIG(sig));
	sinfo = pgpSigToSigInfo( sig );

	if (sinfo->by == NULL)
		return PGP_SIGTRUST_NOKEY;
	if (!(sinfo->trust & PGP_SIGTRUSTF_TRIED))
		return PGP_SIGTRUST_UNTRIED;
	if (!(sinfo->trust & PGP_SIGTRUSTF_CHECKED))
		return PGP_SIGTRUST_BAD;
	else
		return sinfo->trust & kPGPKeyTrust_Mask;
}

	PGPBoolean
pgpSigChecked(PGPKeyDBObj const *sig)
{
	PGPSigInfo *sinfo;

	pgpAssert(OBJISSIG(sig));
	sinfo = pgpSigToSigInfo( sig );

	return sinfo->trust & PGP_SIGTRUSTF_CHECKED;
}

	PGPBoolean
pgpSigTried(PGPKeyDBObj const *sig)
{
	PGPSigInfo *sinfo;

	pgpAssert(OBJISSIG(sig));

	sinfo = pgpSigToSigInfo( sig );
	return sinfo->trust & PGP_SIGTRUSTF_TRIED;
}

	PGPBoolean
pgpSigExportable(PGPKeyDBObj const *sig)
{
	PGPSigInfo *sinfo;

	pgpAssert(OBJISSIG(sig));
	sinfo = pgpSigToSigInfo( sig );

	return SIGISEXPORTABLE(sinfo);
}

	PGPByte
pgpSigTrustLevel(PGPKeyDBObj const *sig)
{
	PGPSigInfo *sinfo;

	pgpAssert(OBJISSIG(sig));
	sinfo = pgpSigToSigInfo( sig );

	return sinfo->trustLevel;
}

	PGPByte
pgpSigTrustValue(PGPKeyDBObj const *sig)
{
	PGPSigInfo *sinfo;

	pgpAssert(OBJISSIG(sig));
	sinfo = pgpSigToSigInfo( sig );

	return sinfo->trustValue;
}

/* Call pgpSigTrust to get sig status, then call this function if
   sig is good and the confidence is required. */

	PGPUInt16
pgpSigConfidence(PGPKeyDBObj *sig)
{
	PGPKeyDBObj *key;
	PGPSigInfo *sinfo;
	PGPKeyInfo *kinfo;

	pgpAssert(OBJISSIG(sig));
	sinfo = pgpSigToSigInfo( sig );

	if (sinfo->by != NULL) {
		key = sinfo->by;
		pgpAssert( OBJISKEY( key ) );
		kinfo = pgpKeyToKeyInfo( key );
		if (kinfo->trust & PGP_KEYTRUSTF_REVOKED)
			return 0;
		else
			return pgpKeyCalcTrust (key);
	}
	else
		return 0;
}


	PGPUInt32
pgpSigType(PGPKeyDBObj const *sig)
{
	PGPSigInfo *sinfo;

	pgpAssert(OBJISSIG(sig));
	sinfo = pgpSigToSigInfo( sig );

	return sinfo->type;
}

	PGPUInt32
pgpSigTimestamp(PGPKeyDBObj const *sig)
{
	PGPSigInfo *sinfo;

	pgpAssert(OBJISSIG(sig));
	sinfo = pgpSigToSigInfo( sig );

	return sinfo->tstamp;
}

	PGPUInt32
pgpSigExpiration(PGPKeyDBObj const *sig)
{
	PGPSigInfo *sinfo;

	pgpAssert(OBJISSIG(sig));
	sinfo = pgpSigToSigInfo( sig );

	if (sinfo->tstamp == 0 || sinfo->sigvalidity == 0)
		return 0;    /* valid indefinitely */
	else
		return sinfo->tstamp + sinfo->sigvalidity;
}

/*
 * True if sig is a self-sig.
 */
	PGPBoolean
pgpSigIsSelfSig(PGPKeyDBObj const *sig)
{
	PGPKeyDBObj const	*parent;
	PGPSigInfo *sinfo;

	pgpAssert(OBJISSIG(sig));
	sinfo = pgpSigToSigInfo( sig );

	/* Find top-level key */
	parent = sig;
	while (!OBJISTOPKEY(parent))
		parent = parent->up;

	/* No good if not signed by top-level key (selfsig) */
	if (sinfo->by != parent)
		return FALSE;

	/* All OK */
	return TRUE;
}



/*
 * True if sig is an X509 sig.
 */
	PGPBoolean
pgpSigIsX509(PGPKeyDBObj const *sig)
{
	PGPSigInfo *sinfo;

	pgpAssert(OBJISSIG(sig));
	sinfo = pgpSigToSigInfo( sig );

	return SIGISX509(sinfo) != 0;
}

	PGPByte *
pgpSigX509Certificate(PGPKeyDBObj *sig, PGPSize *len)
{
	PGPSigInfo *sinfo;
	PGPByte *ptr;

	pgpAssert(OBJISSIG(sig));
	pgpAssert(IsntNull(len));
	sinfo = pgpSigToSigInfo( sig );

	*len = 0;
	if (!SIGISX509 (sinfo))
		return NULL;
	ptr = (PGPByte *)pgpFetchObject(sig, len);
	if (IsNull( ptr ) )
		return NULL;
	ptr = (PGPByte *)pgpSigFindNAISubSubpacket(ptr, SIGSUBSUB_X509, 0, len,
												NULL, NULL, NULL, NULL);
	if( IsntNull( ptr ) ) {
		/* Skip type, version bytes */
		pgpAssert (ptr[0] == SIGSUBSUB_X509);
		ptr += 3;
		*len -= 3;
	}

	return ptr;
}

	PGPBoolean
pgpCRLChecked(PGPKeyDBObj const *crl)
{
	PGPCRLInfo *cinfo;

	pgpAssert(OBJISCRL(crl));
	cinfo = pgpCRLToCRLInfo( crl );

	return (cinfo->trust & PGP_SIGTRUSTF_CHECKED) != 0;
}

	PGPBoolean
pgpCRLTried(PGPKeyDBObj const *crl)
{
	PGPCRLInfo *cinfo;

	pgpAssert(OBJISCRL(crl));
	cinfo = pgpCRLToCRLInfo( crl );

	return (cinfo->trust & PGP_SIGTRUSTF_TRIED) != 0;
}

	PGPUInt32
pgpCRLCreation(PGPKeyDBObj const *crl)
{
	PGPCRLInfo *cinfo;

	pgpAssert(OBJISCRL(crl));
	cinfo = pgpCRLToCRLInfo( crl );

	return cinfo->tstamp;
}

	PGPUInt32
pgpCRLExpiration(PGPKeyDBObj const *crl)
{
	PGPCRLInfo *cinfo;

	pgpAssert(OBJISCRL(crl));
	cinfo = pgpCRLToCRLInfo( crl );

	return cinfo->tstampnext;
}

/* True if the key has a CRL */
	PGPBoolean
pgpKeyHasCRL(PGPKeyDBObj *key)
{
	PGPKeyDBObj *child;

	for (child = key->down; IsntNull(child); child = child->next) {
		if (!OBJISCRL(child))
			continue;
		if( !pgpKeyDBObjIsReal(child) )
			continue;
		if (!pgpCRLChecked(child))
			continue;
		return TRUE;
	}
	return FALSE;
}

/* Return nth CRL of key, along with a count of all CRLs if requested.
 * Doesn't count superceded CRLs.
 * Call this the first time with n=0 and &crlcount, and later times with
 * crlcount holding NULL.
 */
	PGPKeyDBObj *
pgpKeyNthCRL(PGPKeyDBObj *key, PGPUInt32 n, PGPUInt32 *crlcount)
{
	PGPKeyDBObj *obj;
	PGPKeyDBObj *nthcrl = NULL;
	PGPUInt32 count = 0;

	if( IsntNull( crlcount ) )
		*crlcount = 0;

	for (obj = key->down; IsntNull(obj); obj = obj->next) {
		if (!OBJISCRL(obj))
			continue;
		if( !pgpKeyDBObjIsReal(obj) )
			continue;
		if (!pgpCRLChecked(obj))
			continue;
		if (!pgpCRLIsCurrent(obj, 0))
			continue;
		if (count++ == n) {
			nthcrl = obj;
			if (IsNull( crlcount ))
				break;
		}
	}
	if( IsntNull( crlcount ) )
		*crlcount = count;
	return nthcrl;
}


/*
 * Find the earliest non-replaced CRL issued by a key.
 * If expiration is true, use expiration dates, else creation dates.
 */
	PGPKeyDBObj *
pgpKeyEarliestCRL(PGPKeyDBObj *key, PGPBoolean expiration)
{
	PGPKeyDBObj	*crl;
	PGPKeyDBObj *bestcrl = NULL;
	PGPUInt32 crltime;
	PGPUInt32 besttime = (PGPUInt32)-1L;
	PGPUInt32 n = 0;
	PGPUInt32 crlcount = 0;

	do {
		crl = pgpKeyNthCRL(key, n, (n?NULL:&crlcount));
		if( IsNull( crl ) )
			break;
		if( expiration )
			crltime = pgpCRLExpiration( crl );
		else
			crltime = pgpCRLCreation( crl );
		if (crltime < besttime) {
			besttime = crltime;
			bestcrl = crl;
		}
	} while (n++ < crlcount);

	return bestcrl;
}


/*
 * See whether there is a more recent CRL for this key with the same
 * dist. point.
 * If tstamp is nonzero, also checks for expiration of CRL.
 */
	PGPBoolean
pgpCRLIsCurrent (PGPKeyDBObj *crl, PGPUInt32 tstamp)
{
	PGPKeyDBObj *obj;
	PGPKeyDBObj *key;
	PGPKeyDB	*db;
	PGPCRLInfo *cinfo;
	PGPCRLInfo *cinfo2;
	PGPUInt32 crlcreation, crlexpiration;
	PGPUInt32 objcreation;
	PGPByte const *tmpdpoint;
	PGPByte *dpoint = NULL;
	PGPByte const *dpoint2;
	PGPSize dpointlen;
	PGPSize dpoint2len;
	PGPContextRef context;

	pgpAssert (OBJISCRL(crl));
	cinfo = pgpCRLToCRLInfo( crl );

	db = PGPPeekKeyDBObjKeyDB( crl );
	context = PGPPeekKeyDBContext( db );
	
	crlcreation = pgpCRLCreation(crl);
	crlexpiration = pgpCRLExpiration(crl);
	if (tstamp != 0) {
		if (crlexpiration < tstamp)
			return FALSE;
	}

	if( CRLHASDPOINT( cinfo ) ) {
		/* Read dpoint data structure */
		tmpdpoint = pgpCRLDistributionPoint( crl, &dpointlen );
		dpoint = pgpContextMemAlloc( context, dpointlen, 0 );
		pgpCopyMemory( tmpdpoint, dpoint, dpointlen );
		if( IsNull( dpoint ) ) {
			return TRUE;
		}
	}

	/* Find key above CRL */
	key = crl->up;
	pgpAssert (OBJISKEY(key));
	for (obj = key->down; IsntNull(obj); obj = obj->next) {
		pgpAssert(obj);
		if (obj == crl)
			continue;
		if (!OBJISCRL(obj))
			continue;
		if( !pgpKeyDBObjIsReal(obj) )
			continue;
		if (!pgpCRLChecked(obj))
			continue;
		cinfo2 = pgpCRLToCRLInfo( obj );
		if (CRLHASDPOINT(cinfo2) != CRLHASDPOINT(cinfo))
			continue;
		if (CRLHASDPOINT(cinfo)) {
			dpoint2 = pgpCRLDistributionPoint( obj, &dpoint2len );
			if (dpointlen != dpoint2len ||
				!pgpMemoryEqual( dpoint, dpoint2, dpointlen ))
				continue;
		}
		objcreation = pgpCRLCreation(obj);
		if (objcreation > crlcreation) {
			if (CRLHASDPOINT( cinfo ) )
				pgpContextMemFree( context, dpoint );
			return FALSE;
		}
	}
	if (CRLHASDPOINT( cinfo ) )
		pgpContextMemFree( context, dpoint );
	return TRUE;
}

/* Return CRL Distribution Point, in the "fetch" buffer */
/* Return NULL if CRL has no distribution point */
	PGPByte const *
pgpCRLDistributionPoint( PGPKeyDBObj *crl, PGPSize *len )
{
	PGPByte const *ptr;
	PGPCRLInfo *cinfo;

	pgpAssert (OBJISCRL(crl));
	pgpAssert (IsntNull( len ) );
	cinfo = pgpCRLToCRLInfo( crl );

	*len = 0;
	if( !CRLHASDPOINT( cinfo ) ) 
		return NULL;

	ptr = pgpFetchObject(crl, len);
	if (IsNull( ptr ) )
		return NULL;
	ptr = pgpCRLFindDPoint( ptr, *len, len );

	return ptr;
}


/*
 * Return a buffer full of all the CRL Distribution Points associated with
 * a given key.  We find them by listing those in the CRLs we have stored
 * with the key, plus we search all certs which the key has issued and get
 * any others from those.  Also return the number of CDP's in the buffer.
 * The buffer is a newly allocated buffer.
 * Also return a newly allocated buffer of sizes of the CDP's.
 */
	PGPError
pgpListCRLDistributionPoints(
	PGPMemoryMgrRef mgr,				/* Input parameters */
	PGPKeyDBObj *key,
	PGPUInt32 *pnDistPoints,			/* Output parameters */
	PGPByte **pbuf,
	PGPSize **pbufsizes
	)
{
	PGPContextRef context;
	PGPKeyInfo *kinfo;
	PGPSigInfo *sinfo;
	PGPASN_XTBSCertificate *xtbscert;
	PGPUInt32 nDistPoints = 0;
	PGPByte *buf = NULL;
	PGPSize bufsize = 0;
	PGPSize *bufsizes = NULL;
	PGPByte const *crldpoint;
	PGPByte const *certdpoint = NULL;		/* dynamically allocated */
	PGPSize dpointlen;
	PGPError err = kPGPError_NoErr;
	PGPUInt32 nth;
	PGPKeyDBObj *crlobj;
	PGPKeyDBObj *sig;
	PGPByte *p;
	PGPSize len;
	void *vbuf;
	PGPUInt32 crlcount;

	pgpAssert( IsntNull( pbuf ) );
	pgpAssert( IsntNull( pbufsizes ) );
	pgpAssert( IsntNull( pnDistPoints ) );
    pgpAssert (OBJISTOPKEY(key));
	kinfo = pgpKeyToKeyInfo( key );
	context = PGPPeekKeyDBContext(PGPPeekKeyDBObjKeyDB( key ) );

	*pbuf = NULL;
	*pbufsizes = NULL;
	*pnDistPoints = 0;

	/* First check any CRLs stored with the key */
	nth = 0;
	do
	{
		crlobj = pgpKeyNthCRL( key, nth, &crlcount );
		if( IsNull( crlobj ) )
			break;
		crldpoint = pgpCRLDistributionPoint( crlobj, &dpointlen );
		if( IsntNull( crldpoint ) )
		{
			if( !pgpX509BufInSequenceList( crldpoint,dpointlen,buf,bufsize ) )
			{
				bufsize += dpointlen;
				vbuf = buf;
				err = PGPReallocData( mgr, &vbuf, bufsize, 0 );
				buf = vbuf;
				if( IsPGPError( err ) )
					goto error;
				pgpCopyMemory( crldpoint, buf+bufsize-dpointlen, dpointlen );
				++nDistPoints;
				vbuf = bufsizes;
				err = PGPReallocData( mgr, &vbuf,
									  nDistPoints*sizeof(PGPSize), 0 );
				bufsizes = vbuf;
				if( IsPGPError( err ) )
					goto error;
				bufsizes[nDistPoints-1] = dpointlen;
			}
		}
	} while (++nth < crlcount);

	/* Now check all certs signed by key */
	for( sig = kinfo->sigsby; IsntNull( sig ); sig = sinfo->nextby )
	{
		sinfo = pgpSigToSigInfo(sig);
		if( !pgpKeyDBObjIsReal(sig) )
			continue;
		if( !SIGISX509( sinfo )  ||  !SIGHASDISTPOINT( sinfo ) )
			continue;
		/* Find distribution point in cert */
		p = (PGPByte *)pgpFetchObject(sig, &len);
		if( IsNull( p ) )
			continue;
		p = (PGPByte *)pgpSigFindNAISubSubpacket(p, SIGSUBSUB_X509, 0, &len,
												  NULL, NULL, NULL, NULL);
		if( IsNull( p ) )
			continue;
		p += 3;
		len -= 3;
		err = pgpX509BufferToXTBSCert( context, p, len, &xtbscert);
		if( IsPGPError( err ) ) {
			err = kPGPError_NoErr;
			continue;
		}
		/* Note that dpointcert is dynamically allocated */
		certdpoint = pgpX509XTBSCertToDistPoint( xtbscert, &dpointlen );
		pgpX509FreeXTBSCert( context, xtbscert );

		if( IsntNull( certdpoint ) &&
			!pgpX509BufInSequenceList( certdpoint, dpointlen, buf, bufsize ) )
		{
			bufsize += dpointlen;
			vbuf = buf;
			err = PGPReallocData( mgr, &vbuf, bufsize, 0 );
			buf = vbuf;
			if( IsPGPError( err ) )
				goto error;
			pgpCopyMemory( certdpoint, buf+bufsize-dpointlen, dpointlen );
			++nDistPoints;
			vbuf = bufsizes;
			err = PGPReallocData( mgr, &vbuf,
								  nDistPoints*sizeof(PGPSize), 0);
			bufsizes = vbuf;
			if( IsPGPError( err ) )
				goto error;
			bufsizes[nDistPoints-1] = dpointlen;
		}
		PGPFreeData( (PGPByte *)certdpoint );
		certdpoint = NULL;
	}
	
	*pbuf = buf;
	*pbufsizes = bufsizes;
	*pnDistPoints = nDistPoints;

	return kPGPError_NoErr;

 error:

	if( IsntNull( buf ) )
		PGPFreeData( buf );
	if( IsntNull( bufsizes ) )
		PGPFreeData( bufsizes );
	if( IsntNull( certdpoint ) )
		PGPFreeData( (PGPByte *)certdpoint );
	return err;

}

/* Helper function for below - Given a sig, see if it has a subpacket
 * of the type we are interested in.  Given that we have seen *pmatches
 * other subpackets of that type so far, see if this is the nth one.
 * If so, set the various return parameters to point at the data for
 * this subpacket.
 */
static PGPError
sFindSubpacket (PGPKeyDBObj *sig, int subpacktype, unsigned nth,
	PGPByte const **pdata, PGPSize *plen, PGPBoolean *pcritical,
	PGPBoolean *phashed, PGPUInt32 *pcreation, unsigned *pmatches,
	PGPKeyDBObj **psig )
{
	PGPByte *p;
	PGPByte *subp;
	PGPUInt32 matches;
	PGPSize len;

	/* Here we have a nonrevocable self signature */
	p = (PGPByte *)pgpFetchObject(sig, &len);
	if (!p)
		return pgpKeyDBError(PGPPeekKeyDBObjKeyDB(sig));

	subp = (PGPByte *)pgpSigFindSubpacket (p, subpacktype, 0, NULL, NULL,
										   NULL, NULL, &matches);
	if (subp)
	{
		if (nth >= *pmatches && nth < *pmatches+matches)
		{
			/* This packet has the nth instance */
			*psig = sig;
			*pdata = pgpSigFindSubpacket (p, subpacktype, nth - *pmatches,
								plen, pcritical, phashed, pcreation, NULL );
		}
		*pmatches += matches;
	}
	return kPGPError_NoErr;
}



/*
 * Return key self-sig subpacket information.  Searches all sigs below
 * the key for a self sig, finds most recent one with desired info.
 * nth is 0 to find first matching packet, 1 for second, etc.  The
 * semantics have changed (yet again) from earlier versions in order
 * to more easily supercede old signatures.
 * 
 * The rule now is that non-revocable signatures always have precedence.
 * Any subpacket in such a signature cannot be revoked, and will be returned
 * first.  Then, we take the latest self signature on the primary userid
 * of the key and we see whether it has the subpacket of interest.
 * So there is one revocable self signature that counts, and it is the
 * latest one.  Plus, all the nonrevocable self sigs count.
 *
 * pmatches is filled in with the total
 * number of instances in all packets.  The plen, pcritical, phashed,
 * and pcreated values are filled in with the signature packet
 * corresponding to the nth instance of the data we want.
 *
 * key			key to use
 * subpacktype	subpacket type to search for
 * nth			nth matching subpacket to find
 * *plen		return length of data
 * *pcritical	return criticality field of subpacket
 * *phashed		return whether subpacket was in hashed region
 * *pcreation	return creation time of matching signature
 * *pmatches	return number of matches of this subpack type
 * *error		return error code
 *
 * Function returns pointer to the data, of length *plen, or NULL with *error
 * set for error code.  If matching packet is not found, return NULL
 * with *error = 0.
 */
	PGPByte const *
pgpKeyFindSubpacket (PGPKeyDBObj *obj, int subpacktype, unsigned nth,
	PGPSize *plen, PGPBoolean *pcritical, PGPBoolean *phashed,
	PGPUInt32 *pcreation, PGPUInt32 *pmatches, PGPError *error)
{
	PGPKeyDBObj		*sig, *bestsig, *lookobj, *signobj;
	PGPKeyDBObj		*child;
	PGPKeyDBObj		*primaryUserid;
	PGPSigInfo *	sinfo;
	PGPUInt32		bestcreation;
	PGPUInt32		totalmatches;
	PGPByte const	*bestp;
	PGPSize			bestlen;
	PGPBoolean		bestcritical;
	PGPBoolean		besthashed;
	int				level;

	pgpAssert(OBJISKEY(obj));

	signobj = OBJISTOPKEY(obj) ? obj : obj->up;
	primaryUserid = pgpKeyPrimaryUserID( signobj, 0 );
	
	bestlen = bestcritical = besthashed = bestcreation = 0;
	totalmatches = 0;
	sig = NULL;
	bestp = NULL;
	bestsig = NULL;

	if (error)
		*error = kPGPError_NoErr;

	/* First look for nonrevocable signatures */
	for (child = obj->down; IsntNull(child); child = child->next)
	{
		if( !pgpKeyDBObjIsReal( child ) )
			continue;
		/* Kludge here to look for sigs at either level 2 or level 3 */
		if( OBJISSIG( child ) )
		{
			sig = child;
			level = 2;
		} else {
			sig = child->down;
			level = 3;
		}
		/* This does one pass for level 2 sigs, or multiples for level 3 */
		for ( ; IsntNull( sig ) ; sig = (level == 2) ? NULL : sig->next )
		{
			/* Abort when we come to another key or subkey. */
			if (OBJISKEY(sig))
				goto done1;
			if( !OBJISSIG(sig))
				continue;
			if( !pgpKeyDBObjIsReal(sig) )
				continue;
			/* Only count self-sigs that have been validated */
			if (pgpSigMaker (sig) != signobj)
				continue;
			if ((pgpSigType (sig) & 0xf0) != PGP_SIGTYPE_KEY_GENERIC)
				continue;
			if (!pgpSigChecked (sig))
				continue;
			sinfo = pgpSigToSigInfo( sig );
			if (SIGISREVOCABLE(sinfo))
				continue;

			/* Here we have a nonrevocable self signature */
			*error = sFindSubpacket( sig, subpacktype, nth, &bestp, &bestlen,
									 &bestcritical, &besthashed, &bestcreation,
									 &totalmatches, &bestsig );
			if( IsPGPError( *error ) )
				return NULL;

			/* If don't need to count all matches, done now */
			if( IsNull( pmatches ) && IsntNull( bestsig ) )
				break;
		}
	}
done1:

	/* Second, on subkey, look below it; on key, look below primary */
	if( OBJISSUBKEY(obj) )
		lookobj = obj;
	else
		lookobj = primaryUserid;
	/* Only check if we don't have bestsig yet or we are returning matches */
	if (IsntNull(lookobj) && (IsNull(bestsig) || IsntNull(pmatches)))
	{
		sig = pgpLatestSigByKey( lookobj, signobj );
		if (sig)
		{
			sinfo = pgpSigToSigInfo( sig );
			if (SIGISREVOCABLE(sinfo))
			{
				*error = sFindSubpacket( sig, subpacktype, nth, &bestp,
										 &bestlen, &bestcritical, &besthashed,
										 &bestcreation, &totalmatches,
										 &bestsig );
				if( IsPGPError( *error ) )
					return NULL;
			}
		}
	}

	/* Third, look at other userids */
	/* Only check if we don't have bestsig yet or we are returning matches */
	if (IsntNull(lookobj) && (IsNull(bestsig) || IsntNull(pmatches)))
	{
		for (lookobj = signobj->down; IsntNull(lookobj); lookobj=lookobj->next)
		{
			if( !pgpKeyDBObjIsReal( lookobj ) )
				continue;
			if( !OBJISUSERID( lookobj ) )
				continue;
			if( lookobj == primaryUserid )
				continue;
			sig = pgpLatestSigByKey( lookobj, signobj );
			if (sig)
			{
				sinfo = pgpSigToSigInfo( sig );
				if (SIGISREVOCABLE(sinfo))
				{
					*error = sFindSubpacket( sig, subpacktype, nth, &bestp,
											 &bestlen, &bestcritical,
											 &besthashed, &bestcreation,
											 &totalmatches, &bestsig );
					if( IsPGPError( *error ) )
						return NULL;
				}
			}
			/* If don't need to count all matches, done now */
			if( IsNull( pmatches ) && IsntNull( bestsig ) )
				break;
		}
	}

	/* Fourth, check sigs directly below signing key */
	/* Only check if we don't have bestsig yet or we are returning matches */
	if( OBJISKEY( signobj )  &&  (IsNull(bestsig) || IsntNull(pmatches)))
	{
		lookobj = signobj;

		sig = pgpLatestSigByKey( lookobj, signobj );
		if (sig)
		{
			sinfo = pgpSigToSigInfo( sig );
			if (SIGISREVOCABLE(sinfo))
			{
				*error = sFindSubpacket( sig, subpacktype, nth, &bestp,
										 &bestlen, &bestcritical, &besthashed,
										 &bestcreation, &totalmatches,
										 &bestsig );
				if( IsPGPError( *error ) )
					return NULL;
			}
		}
	}

	if (plen)
		*plen = bestlen;
	if (pcritical)
		*pcritical = bestcritical;
	if (phashed)
		*phashed = besthashed;
	if (pcreation)
		*pcreation = bestcreation;
	if (pmatches)
		*pmatches = totalmatches;
	return bestp;
}


/*
 * Find an additional decryption key for the given key, if one exists.
 * nth tells which one to find.  *pkeys is set to the number of add'l
 * decryption keys, *pclass is set to the class byte associated with
 * the decryption key.  *pkalg and *keyid are set to the algorithm and
 * keyid of the nth ADK key.  Returns NULL but no error in *error if
 * the ADK key is not found.  Return *error as
 * kPGPError_ItemNotFound if there are fewer than n+1 ADKs.
 */
	PGPKeyDBObj *
pgpKeyAdditionalRecipientRequestKey (PGPKeyDBObj *obj,
	unsigned nth, PGPByte *pkalg, PGPKeyID *keyid,
	PGPByte *pclass, unsigned *pkeys, PGPError *error)
{
	PGPKeyDBObj	   *rkey;			/* Additional decryption key */
	PGPByte const  *krpdata;		/* Pointer to key decryption data */
	PGPSize			krdatalen;		/* Length of krdata */
	PGPBoolean		critical;		/* True if decryption field was critical */
	PGPBoolean		hashed;			/* True if was in hashed region */
	unsigned		matches;		/* Number of adk's found */
	PGPByte			fingerp[20];	/* Fingerprint of adk */
	PGPByte			krdata[22];		/* Copy of key decryption data packet */

	pgpAssert(OBJISKEY(obj));
	pgpAssert (error);

	*error	= kPGPError_NoErr;
	if( IsntNull( pkeys ) )
		*pkeys	= 0;
	if( IsntNull( pclass ) )
		*pclass	= 0;
	if( IsntNull( pkalg ) )
		*pkalg = 0;
	if( IsntNull( keyid ) )
	{
		pgpClearMemory( keyid, sizeof( *keyid ) );
	}
	
	krpdata = pgpKeyFindSubpacket (obj,
		SIGSUB_KEY_ADDITIONAL_RECIPIENT_REQUEST, nth, &krdatalen,
		&critical, &hashed, NULL, &matches, error);
	if (!krpdata  ||  !hashed) {
		if (IsntPGPError(*error))
			*error = kPGPError_ItemNotFound;
		return NULL;
	}
	/*
	 * krdata is 1 byte of class, 1 of pkalg, 20 bytes of fingerprint.
	 * Last 8 of 20 are keyid.  Make a copy because data is volatile when
	 * we do other operations.
	 */

	if (krdatalen < sizeof(krdata)) {
		/* malformed packet, can't use it */
		*error = kPGPError_ItemNotFound;
		return NULL;
	}
	pgpCopyMemory (krpdata, krdata, sizeof(krdata));

	/* Do we have ADK? */
	rkey = pgpKeyById8 (PGPPeekKeyDBObjKeyDB(obj), krdata[1],krdata+2+20-8);
	if (IsntNull (rkey)) {
		if( pgpKeyDBObjIsDummy( rkey ) ) {
			rkey = NULL;
		} else {
			pgpKeyFingerprint20 (rkey, fingerp);
			if (memcmp (fingerp, krdata+2, 20) != 0) {
				/* Have a key that matches in keyid but wrong fingerprint */
				rkey = NULL;
			}
		}
	}
	/* Success */
	if (pkeys) {
		*pkeys = matches;
	}
	if (pclass) {
		*pclass = krdata[0];
	}
	if (pkalg) {
		*pkalg = krdata[1];
	}
	if (keyid) {
		pgpNewKeyIDFromRawData( krdata+2+20-8, krdata[1], 8, keyid );
	}
	return rkey;
}

	
/*
 * Find a key revocation key and keyid for the given key.  nth tells
 * which one to find.  *pkeys is set to the number of key revocation
 * keys, *pclass is set to the class byte associated with the
 * revocation key.  Returns NULL but no error in *error if the ADK key
 * is not found.  Return *error as
 * kPGPError_ItemNotFound if there are fewer than n+1 ADKs.  The class
 * byte is intended for future expansion; for now the high order bit
 * is used to indicate a revocation authorization.  Later we could use
 * the other bits to authorize other kinds of signatures, perhaps.
 */
	PGPKeyDBObj *
pgpKeyRevocationKey (PGPKeyDBObj *obj, unsigned nth,
	PGPByte *pkalg, PGPKeyID *keyid, PGPByte *pclass, unsigned *pkeys,
	PGPError *error)
{
	PGPKeyDBObj	   *rkey;			/* Message revocation key */
	PGPByte const  *krpdata;		/* Pointer to key revocation data */
	PGPSize			krdatalen;		/* Length of krdata */
	PGPBoolean		critical;		/* True if revocation field was critical */
	PGPBoolean		hashed;			/* True if in hashed region (must be) */
	unsigned		matches;		/* Number of revkey packets found */
	PGPByte			fingerp[20];	/* Fingerprint of revkey */
	PGPByte			krdata[22];		/* Copy of key revocation data packet */

	pgpAssert(OBJISKEY(obj));
	pgpAssert (error);

	*error	= kPGPError_NoErr;
	if( IsntNull( pkeys ) )
		*pkeys	= 0;
	if( IsntNull( pclass ) )
		*pclass	= 0;
	if( IsntNull( pkalg ) )
		*pkalg = 0;
	if( IsntNull( keyid ) )
	{
		pgpClearMemory( keyid, sizeof( *keyid ) );
	}
	
	krpdata = pgpKeyFindSubpacket (obj, SIGSUB_KEY_REVOCATION_KEY, nth,
		&krdatalen, &critical, &hashed, NULL, &matches, error);
	if (!krpdata || !hashed) {
		if (IsntPGPError(*error))
			*error = kPGPError_ItemNotFound;
		return NULL;
	}
	/*
	 * krdata is 1 byte of class, 1 of pkalg, 20 bytes of fingerprint.
	 * Last 8 of 20 are keyid.  Make a copy because data is volatile when
	 * we do other operations.
	 */

	if (krdatalen < sizeof(krdata)) {
		/* malformed packet, can't use it */
		*error = kPGPError_ItemNotFound;
		return NULL;
	}
	pgpCopyMemory (krpdata, krdata, sizeof(krdata));

	/* Do we have revocation packet? */
	rkey = pgpKeyById8 (PGPPeekKeyDBObjKeyDB(obj), krdata[1],krdata+2+20-8);
	if (IsntNull (rkey) ) {
		if (pgpKeyDBObjIsDummy( rkey ) ) {
			rkey = NULL;
		} else {
			pgpKeyFingerprint20 (rkey, fingerp);
			if (memcmp (fingerp, krdata+2, 20) != 0) {
				/* Have a key that matches in keyid but wrong fingerprint */
				rkey = NULL;
			}
		}
	}
	/* Else success */
	if (pkeys) {
		*pkeys = matches;
	}
	if (pclass) {
		*pclass = krdata[0];
	}
	if (pkalg) {
		*pkalg = krdata[1];
	}
	if (keyid) {
		pgpNewKeyIDFromRawData( krdata+2+20-8, krdata[1], 8, keyid );
	}
	return rkey;
}


/*
 * Return true if rkey is a revocation key for key.
 * Includes case where rkey is a dummy key which has issued a signature
 * but is not local to the keyring.  In that case we just check for match
 * on keyid and pkalg.
 */
	PGPBoolean
pgpKeyIsRevocationKey (PGPKeyDBObj *key, PGPKeyDBObj *rkey)
{
	PGPByte				 revClass;
	PGPUInt32			 nRevKeys;
	PGPUInt32			 iRevKeys;
	PGPByte				 revAlg;
	PGPByte				 rKeyAlg;
	PGPKeyID			 revKeyID;
	PGPKeyID			 rKeyID;
	PGPBoolean			 nonLocal;
	PGPError			 error;

	nonLocal = pgpKeyDBObjIsDummy(rkey);
	if (nonLocal) {
		pgpKeyID8 (rkey, &rKeyAlg, &rKeyID);
	}

	nRevKeys = 1;
	iRevKeys = 0;
	while (iRevKeys < nRevKeys) {
		PGPKeyDBObj const *revkey;
		revkey = pgpKeyRevocationKey(key, iRevKeys++, &revAlg, &revKeyID,
									  &revClass, &nRevKeys, &error);
		if (IsPGPError(error))
			break;
		if (!(revClass & 0x80))
			continue;
		if (nonLocal) {
			if (rKeyAlg == revAlg &&
				0 == PGPCompareKeyIDs (&rKeyID, &revKeyID))
				return TRUE;
		} else if (revkey == rkey) {
			return TRUE;
		}
	}
	return FALSE;
}
	

/*
 * True if there is a third party revocation signature on the given
 * key.  Return the key (if local) and the keyid of the revoking key.
 * Note that this requires that there be both a revoking authorization
 * on the revoker key and a revocation issued by that key.
 */
	PGPBoolean
pgpKeyHasThirdPartyRevocation (PGPKeyDBObj *obj,
	PGPKeyDBObj **revkey, PGPByte *pkalg, PGPKeyID *keyid, PGPError *error)
{
	PGPSigInfo		*sinfo = NULL;
	PGPKeyDBObj		*topkey;
	PGPKeyDBObj		*sig;
	PGPByte			 rKeyAlg;
	PGPKeyID		 rKeyID;

	pgpAssert(OBJISKEY(obj));
	pgpAssert (error);

	*error	= kPGPError_NoErr;
	if( IsntNull( revkey ) )
		*revkey = NULL;
	if( IsntNull( pkalg ) )
		*pkalg = 0;
	if( IsntNull( keyid ) )
	{
		pgpClearMemory( keyid, sizeof( *keyid ) );
	}

	/* May be called on subkey, so find top level key */
	topkey = obj;
	if (OBJISSUBKEY(topkey))
		topkey = pgpKeyMasterkey (topkey );
	
	/* Search for a revocation signature on this key */
	for (sig=obj->down; IsntNull(sig); sig=sig->next) {
		if( !pgpKeyDBObjIsReal(sig) )
			continue;
		if (!OBJISSIG(sig))
			continue;
		sinfo = pgpSigToSigInfo( sig );
		if (sinfo->type != ( (topkey==obj) ? PGP_SIGTYPE_KEY_REVOKE
										   : PGP_SIGTYPE_KEY_SUBKEY_REVOKE ))
			continue;
		if (sinfo->by == topkey)
			continue;

		if (pgpKeyIsRevocationKey (topkey, sinfo->by))
			break;
	}

	if (IsNull (sig)) {
		/* No luck */
		return FALSE;
	}

	if( IsntNull( revkey ) ) {
		if( !pgpKeyDBObjIsDummy( sinfo->by ) )
			*revkey = sinfo->by;
	}

	pgpKeyID8 (sinfo->by, &rKeyAlg, &rKeyID);
	
	if (pkalg) {
		*pkalg = rKeyAlg;
	}
	if (keyid) {
		*keyid = rKeyID;
	}
	return TRUE;
}


/*
 * Create a SigSpec structure which represents the given signature.
 * We must have the signing key as a private key.  We would have to
 * change the initialization of pgpSigSpec to avoid this, but normally
 * the reason for doing this is to re-issue a modified signature, so it
 * is a reasonable restriction.
 */
	PGPSigSpec *
pgpSigSigSpec (PGPKeyDBObj *sig, PGPError *error)
{
	PGPContextRef cdkContext;
	PGPEnv *env;
	PGPSigInfo *sinfo;
	PGPByte *p;
	PGPSigSpec *spec;
	PGPSize len;
	PGPSize hlen;

	pgpAssert(OBJISSIG(sig));
	pgpAssert( IsntNull( error ) );
	*error = kPGPError_NoErr;

	sinfo = pgpSigToSigInfo( sig );
	cdkContext = PGPPeekKeyDBContext(PGPPeekKeyDBObjKeyDB( sig ) );
	env = pgpContextGetEnvironment( cdkContext );

	spec = pgpSigSpecCreate (env, sinfo->by, sinfo->type);
	if( IsNull( spec ) ) {
		*error = kPGPError_OutOfMemory;
		return NULL;
	}

	/* Fetch data for sig */
	p = (PGPByte *)pgpFetchObject(sig, &len);
	if( IsNull( p ) ) {
		pgpSigSpecDestroy (spec);
		*error = pgpKeyDBError( PGPPeekKeyDBObjKeyDB(sig) );
		return NULL;
	}

	hlen = pgpPktBufferHeaderLen( p );
	pgpSigSpecSetHashtype (spec, sinfo->hashalg);
	pgpSigSpecSetVersion (spec, p[hlen]);
	pgpSigSpecSetTimestamp (spec, sinfo->tstamp);

	/* Incorporate data from subpackets if present */
	if (pgpSigSpecVersion(spec) == PGPVERSION_4) {
		*error = pgpSigSubpacketsSpec (spec, p);
		if (IsPGPError(*error)) {
			pgpSigSpecDestroy (spec);
			return NULL;
		}
	}
	return spec;
}


static PGPBoolean
sObjIsExpiring( PGPKeyDBObj *obj, PGPTime time1, PGPTime time2 )
{
	PGPKeyInfo *kinfo;
	PGPSigInfo *sinfo;
	PGPTime		  exptime;
	int objtype = pgpObjectType( obj );

	if( objtype == RINGTYPE_KEY ) {
		kinfo = pgpKeyToKeyInfo( obj );
		if (kinfo->creationtime && kinfo->validityperiod) {
			exptime = kinfo->creationtime + kinfo->validityperiod*60*60*24;
			if (exptime >= time1 && exptime < time2) {
				return TRUE;
			}
		}
	} else if( objtype == RINGTYPE_SIG ) {
		sinfo = pgpSigToSigInfo( obj );
		if( sinfo->tstamp && sinfo->sigvalidity ) {
			exptime = sinfo->tstamp + sinfo->sigvalidity;
			if (exptime >= time1 && exptime < time2) {
				return TRUE;
			}
		}
	}
	return FALSE;
}

/*
 * Return TRUE as a conservative answer in case of an error.
 */
	PGPBoolean
pgpKeyDBHasExpiringObjects(PGPKeyDB *db, PGPTime time1, PGPTime time2 )
{
	PGPKeyDBObj *	key;
	PGPKeyDBObj *	child;
	PGPKeyDBObj *	gchild;

	for (key = db->firstKeyInDB; IsntNull(key); key = key->next) {
		if( !pgpKeyDBObjIsReal( key ) )
			continue;
		if( sObjIsExpiring( key, time1, time2 ) )
			return TRUE;
		for (child = key->down; IsntNull(child); child = child->next) {
			if( !pgpKeyDBObjIsReal( child ) )
				continue;
			if( sObjIsExpiring( child, time1, time2 ) )
				return TRUE;
			for (gchild = child->down; IsntNull(gchild);
				 								gchild = gchild->next) {
				if( !pgpKeyDBObjIsReal( gchild ) )
					continue;
				if( sObjIsExpiring( gchild, time1, time2 ) )
					return TRUE;
			}
		}
	}
	return FALSE;
}


/******************* Originally from pgpRngPriv.c **********************/


/*
 * Find a key given a keyID.
 *
 * ViaCrypt added pkalgs 2 and 3 which are limited RSA, but doesn't
 * completely distinguish beterrn them, so this doesn't either.  Sigh.
 */
	PGPKeyDBObj *
pgpKeyDBFindKey(PGPKeyDB const *db, PGPByte pkalg, PGPByte const keyID[8])
{
	PGPKeyInfo *kinfo;
	PGPKeyDBObj *key;

	if( pkalg == 2  ||  pkalg == 3 )
		pkalg = 1;
	for( key = db->firstKeyInDB; IsntNull(key); key = key->next ) {
		if( !pgpKeyDBObjIsReal( key ) )
			continue;
		pgpAssert (OBJISKEY(key));
		kinfo = pgpKeyToKeyInfo( key );
		if( memcmp(keyID, kinfo->keyID, 8) == 0 ) {
			if( pkalg == kinfo->pkalg )
				break;
			/* Cope with ViaCrypt's things */
			if( pkalg == 1 && (kinfo->pkalg == 2  ||  kinfo->pkalg == 3) )
				break;
		}
	}

	return key;
}


/*
 * Find a key given a "20n" fingerprint (SHA-1 hash over numeric data)
 * Note that this does disk accesses and may change RingFile pointers.
 */
	PGPKeyDBObj *
pgpKeyDBFindKey20n(PGPKeyDB *db, PGPByte const *fp20n)
{
	PGPKeyInfo *kinfo;
	PGPKeyDBObj *key;
	PGPByte hashbuf[20];

	for( key = db->firstKeyInDB; IsntNull(key); key = key->next ) {
		/* Look for non-dummy key which matches fingerprint */
		if( !pgpKeyDBObjIsReal( key ) )
			continue;
		pgpAssert (OBJISKEY(key));
		kinfo = pgpKeyToKeyInfo( key );
		if (kinfo->fp20n == fp20n[0]) {
			pgpKeyFingerprint20n (key, hashbuf);
			pgpAssert (hashbuf[0] == kinfo->fp20n);
			if (memcmp (fp20n, hashbuf, sizeof(hashbuf)) == 0)
				break;
		}
	}
	return (PGPKeyDBObj *)key;
}


/*
 * Recompute the sigsby lists.  This also establishes the extra
 * invariant (used in trust propagation) that all signatures by one
 * key on another object are adjacent on that key's sigsby list.
 */
void
pgpKeyDBListSigsBy(PGPKeyDB *db)
{
	PGPKeyDBObj *key;
	PGPKeyDBObj *child;
	PGPKeyDBObj *gchild;
	PGPKeyDBObj *keyby;
	PGPKeyInfo	*kinfo;
	PGPSigInfo	*sinfo;

	/* Initialize sigsby lists to null */
	for (key = db->firstKeyInDB; IsntNull(key); key = key->next) {
		/* Preclear for all key types as they are all pointed to by sigs */
		pgpAssert(OBJISTOPKEY(key));
		kinfo = pgpKeyToKeyInfo( key );
		kinfo->sigsby = NULL;
	}

	/* Install every sig on a sigsby list */
	for (key = db->firstKeyInDB; IsntNull(key); key = key->next) {
		if( !pgpKeyDBObjIsReal( key ) )
			continue;
		for (child = key->down; IsntNull(child); child = child->next) {
			if( !pgpKeyDBObjIsReal( child ) )
				continue;
	        if (OBJISSIG(child)) {
				sinfo = pgpSigToSigInfo( child );
				keyby = sinfo->by;
				kinfo = pgpKeyToKeyInfo( keyby );
				sinfo->nextby = kinfo->sigsby;
				kinfo->sigsby = child;
			} else {
				for (gchild = child->down; IsntNull(gchild);
					 							gchild = gchild->next) {
					if( !pgpKeyDBObjIsReal( gchild ) )
						continue;
					if (OBJISSIG(gchild)) {
						sinfo = pgpSigToSigInfo( gchild );
						keyby = sinfo->by;
						kinfo = pgpKeyToKeyInfo( keyby );
						sinfo->nextby = kinfo->sigsby;
						kinfo->sigsby = gchild;
					}
				}
			}
	    }
	}
}


/*
 * Return TRUE if the specified subkey has a valid sig from the main key.
 * Assumes subkey sigs are always tried, which should happen when they are
 * created or added to the keyring.  The only time this isn't true is when
 * we are considering adding a key.  We will give the sig the benefit of
 * the doubt in that case as we aren't using it yet.
 */
	PGPBoolean
pgpSubkeyValid(PGPKeyDBObj *subkey, PGPBoolean unExpired, PGPBoolean revokedOK)
{
	PGPKeyDBObj *sig;
	PGPKeyDBObj *key;
	PGPKeyInfo *kinfo;
	PGPUInt32		  curtime;
	PGPUInt32		  exptime;

	pgpAssert(OBJISSUBKEY(subkey));
	key = subkey->up;
	pgpAssert(OBJISTOPKEY(key));

	kinfo = pgpKeyToKeyInfo( subkey );
	if (!revokedOK && (kinfo->trust & PGP_KEYTRUSTF_REVOKED))
	    return 0;
	if (unExpired) {
		/* Don't use key if has expired or creation time is > 24 hours
			in future */
		if (kinfo->trust & PGP_KEYTRUSTF_EXPIRED)
			return 0;
		curtime = (PGPUInt32) PGPGetTime();
		exptime = (PGPUInt32) pgpKeyExpiration(subkey);
		if ((exptime != 0  &&  curtime > exptime)  ||
			curtime < pgpKeyCreation(subkey) - 24*60*60)
			return 0;
	}
	/* Check legality of subkey */
	for (sig = subkey->down; IsntNull(sig); sig = sig->next) {
		if( !pgpKeyDBObjIsReal(sig) )
			continue;
		if (OBJISSIG(sig) &&
		    pgpSigMaker(sig)==key &&
		    pgpSigType(sig) == PGP_SIGTYPE_KEY_SUBKEY) {
			if (!pgpSigTried(sig))
				return 1; /* could check it here... */
			if (pgpSigChecked(sig))
				return 1;
		}
	}
	return 0;
}



/*
 * Do a fingerprint20 (SHA-1) hash on the specified buffer, which
 * should be key data.  We prefix it with the type and length bytes
 * for compatibility with key signature hashes (once they become SHA
 * based).  Return the number of bytes in the hash, or negative on
 * error.
 */
	PGPError
pgpFingerprint20HashBuf(PGPContextRef context, PGPByte const *buf, size_t len,
						PGPByte *hash)
{
	PGPHashVTBL const *h;
	PGPHashContext *hc;
	PGPByte tmpbuf[3];
	PGPByte const *p;
	MemPool *pool = pgpPeekContextMemPool( context );
	MemPool cut = *pool;
    PGPMemoryMgrRef	mgr;

	h = pgpHashByNumber (kPGPHashAlgorithm_SHA);
	if (!h)
		return kPGPError_BadHashNumber;

	mgr = pgpPeekContextMemPoolMgr( context );
	hc = pgpHashCreate( mgr, h);
	if (!hc)
	{
		memPoolCutBack( pool, &cut );
		return kPGPError_OutOfMemory;
	}

	/* We use this format even for subkeys */
	tmpbuf[0] = PKTBYTE_BUILD(PKTBYTE_PUBKEY, 1);
	tmpbuf[1] = (PGPByte)(len>>8);
	tmpbuf[2] = (PGPByte)len;
	PGPContinueHash(hc, tmpbuf, 3);
	PGPContinueHash(hc, buf, len);
	p = (PGPByte *) pgpHashFinal(hc);
	memcpy(hash, p, h->hashsize);
	PGPFreeHashContext(hc);
	memPoolCutBack( pool, &cut );
	return h->hashsize;
}

/* Call this for sigs known to use a regexp, to return the regexp.  Loads
 * from disk if necessary.  Returns NULL on error.
 */
	void *
pgpSigRegexp( PGPKeyDBObj const *sig )
{
	PGPKeyDBRef db;
	PGPContextRef context;
	PGPSigInfo *sinfo;
	PGPByte *buf;
	PGPSize len;
	regexp *rexp;
	char const *packet;

	pgpAssert( OBJISSIG( sig ) );
	sinfo = pgpSigToSigInfo( sig );
	pgpAssert( SIGUSESREGEXP( sinfo ) );

	db = PGPPeekKeyDBObjKeyDB( (PGPKeyDBObj *)sig );
	context = PGPPeekKeyDBContext( db );

	if( IsNull(sinfo->regexp) ) {
		/* Here we must load the regexp */

		buf = (PGPByte *)pgpFetchObject(sig, &len);
		if( !buf )
			return NULL;
		/* Note that this may alter the contents of buf */
		packet = (char *)pgpSigFindSubpacket(buf, SIGSUB_REGEXP, 0, &len,
											  NULL, NULL, NULL, NULL);
		pgpAssert( packet );
		if (IsPGPError( pgpRegComp( context, packet, &rexp ) ) )
			return NULL;
		sinfo->regexp = (void *) rexp;
	}
	return sinfo->regexp;
}

	
/*
 * Return true if sig is a valid revocation signature.  It must either be
 * a self sig, or it must be by a revocation key.  May be on either a subkey
 * or main key.  This does not check for expirations.
 */
	PGPBoolean
sigRevokesKey (PGPKeyDBObj *sig)
{
	PGPSigInfo *			sinfo;
	PGPKeyInfo *			kinfo;
	PGPKeyDBObj * 			parent;
	PGPKeyDBObj * 			top;

	pgpAssert (OBJISSIG (sig));
	if( !pgpKeyDBObjIsReal( sig ) )
		return FALSE;

	sinfo = pgpSigToSigInfo( sig );
	
	parent = sig->up;
	if (!OBJISKEY(parent))
		return FALSE;
	top = parent;
	while (!OBJISTOPKEY(top))
		top = top->up;

	if (sinfo->type != ( (top==parent) ? PGP_SIGTYPE_KEY_REVOKE
						 			   : PGP_SIGTYPE_KEY_SUBKEY_REVOKE ))
		return FALSE;
	if ((sinfo->trust & PGP_SIGTRUSTF_TRIED) &&
		!(sinfo->trust & PGP_SIGTRUSTF_CHECKED))
		return FALSE;
	/*
	 * If untried, don't accept on a subkey.  Accept on a top-level key
	 * if already shown as revoked because we don't store trust packets
	 * on such keys, so as not to break PGP 2.X.
	 */
	kinfo = pgpKeyToKeyInfo( parent );
	if (!(sinfo->trust & PGP_SIGTRUSTF_TRIED) &&
		((top != parent) ||
		 !(kinfo->trust & PGP_KEYTRUSTF_REVOKED)))
		return FALSE;

	if (sinfo->by == top)
		return TRUE;

	/*
	 * Here we have a revocation signature which is valid but is by some
	 * other key.  We will accept it only if that is a key which is marked
	 * as a revocation authorization key by this one.
	 */

	return pgpKeyIsRevocationKey (top, sinfo->by);
}



/* Look up key by 8 byte keyid and pkalg */
	PGPKeyDBObj *
pgpKeyById8(
	PGPKeyDBRef			db,
	PGPByte				pkalg,
	PGPByte const *		keyIDbytes)
{
	PGPKeyID			keyID;
	PGPKeyDBObj *		key;
	PGPByte				vpkalg;

	vpkalg = pkalg;			/* Viacrypt correction */
	if ((vpkalg | 1) == 3)
		vpkalg = 1;
	pgpNewKeyIDFromRawData( keyIDbytes, vpkalg, 8, &keyID );
	pgpGetKeyByKeyID( db, &keyID, TRUE, FALSE, &key );
	return key;
}


/* use when lookup keyid is < 8 bytes so we have to walk both sides of tree */
	static PGPError
sLookupKeyIDRecursive(
	PGPKeyDBObjRef			key,
	PGPKeyID const *		keyIDIn,
	PGPBoolean				dummyOK,
	PGPBoolean				deletedOK,
	PGPKeyDBObjRef *		outRef )
{
	PGPKeyID		keyIDBytes;
	PGPInt32		cmp;
	PGPKeyInfo *	kinfo;

	if( IsntNull( key ) )
	{
		pgpKeyID8( key, NULL, &keyIDBytes );
		cmp = PGPCompareKeyIDs( &keyIDBytes, keyIDIn);
		if( cmp == 0 )
		{
			if( !deletedOK && pgpKeyDBObjIsDeleted( key ) )
				return kPGPError_ItemNotFound;
			if( !dummyOK && pgpKeyDBObjIsDummy( key ) )
				return kPGPError_ItemNotFound;
			*outRef = key;
			return kPGPError_NoErr;
		}
		kinfo = pgpKeyToKeyInfo( key );
		if( IsntNull( kinfo->left ) )
		{
			if( IsntPGPError( sLookupKeyIDRecursive( kinfo->left, keyIDIn,
													 dummyOK, deletedOK,
													 outRef ) ) )
				return kPGPError_NoErr;
		}
		if( IsntNull( kinfo->right ) )
		{
			if( IsntPGPError( sLookupKeyIDRecursive( kinfo->right, keyIDIn,
													 dummyOK, deletedOK,
													 outRef ) ) )
				return kPGPError_NoErr;
		}
	}
	return kPGPError_ItemNotFound;
}


/* Internal function to look up by keyid */
	PGPError 
pgpGetKeyByKeyID_internal(
	PGPKeyDBRef				db,
	PGPKeyID const *		keyIDIn,
	PGPBoolean				dummyOK,
	PGPBoolean				deletedOK,
	PGPKeyDBObjRef *		outRef )
{
	PGPKeyDBObj *	key;
	PGPKeyID		keyIDBytes;
	PGPInt32		cmp;
	PGPKeyInfo *	kinfo;
	
	PGPValidatePtr( outRef );
	*outRef	= NULL;
	PGPValidateKeyDB( db );
	PGPValidatePtr( keyIDIn );

	key = db->treeKey;

	/* Need special code if looking up with a short keyid */
	if( pgpKeyIDLength( keyIDIn ) < 8 )
		return sLookupKeyIDRecursive( key, keyIDIn, deletedOK, dummyOK,
									  outRef );

	while( IsntNull( key ) )
	{
		pgpKeyID8( key, NULL, &keyIDBytes );
		cmp = PGPCompareKeyIDs( &keyIDBytes, keyIDIn);
		if( cmp == 0 )
		{
			if( !deletedOK && pgpKeyDBObjIsDeleted( key ) )
				return kPGPError_ItemNotFound;
			if( !dummyOK && pgpKeyDBObjIsDummy( key ) )
				return kPGPError_ItemNotFound;
			if( dummyOK && pgpKeyDBObjIsX509Dummy( key ) )
			{
				/* X.509 dummies can point at real keys */
				kinfo = pgpKeyToKeyInfo( key );
				if( IsntNull( kinfo->data ) )
				{
					key = (PGPKeyDBObj *)kinfo->data;
					while( !OBJISTOPKEY( key ) )
						key = key->up;
				}
			}
			*outRef = key;
			return kPGPError_NoErr;
		}
		kinfo = pgpKeyToKeyInfo( key );
		key = (cmp<0) ? kinfo->left : kinfo->right;
	}
	return kPGPError_ItemNotFound;
}


	PGPError 
pgpGetKeyByKeyID(
	PGPKeyDBRef				db,
	PGPKeyID const *		keyIDIn,
	PGPBoolean				dummyOK,
	PGPBoolean				deletedOK,
	PGPKeyDBObjRef *		outRef )
{
	PGPUInt32 id;
	PGPUserValue idobj;
	PGPError err = kPGPError_NoErr;

	if( pgpFrontEndKeyDB( db ) ) {
		/* Front end, call back end */
		PGPContextRef context = PGPPeekKeyDBContext( db );
		PGPValidatePtr( outRef );
		*outRef	= NULL;
		err = pgpGetKeyByKeyID_back( context, db->id, keyIDIn, dummyOK,
									 deletedOK, &id );
		if( IsPGPError( err ) )
			return err;
		if( IsPGPError( PGPFindNode( db->idToObj, id, &idobj ) ) )
			return kPGPError_ItemNotFound;
		*outRef = (PGPKeyDBObj *) idobj;
		return kPGPError_NoErr;
	} else {
		/* Back end, call directly */
		return pgpGetKeyByKeyID_internal( db, keyIDIn, dummyOK,
										  deletedOK, outRef );
	}
}

	PGPError 
PGPFindKeyByKeyID(
	PGPKeyDBRef				keydb,
	PGPKeyID const *		keyIDIn,
	PGPKeyDBObjRef *		outRef )
{
	PGPError	err	= kPGPError_NoErr;
	
	PGPValidatePtr( outRef );
	*outRef	= NULL;
	PGPValidateKeyDB( keydb );
	PGPValidatePtr( keyIDIn );
	
	pgpEnterPGPErrorFunction();

	err	= pgpGetKeyByKeyID( keydb, keyIDIn, TRUE, FALSE, outRef );
	if ( IsntPGPError( err ) )
	{
		if( pgpKeyDBObjIsDummy( *outRef ) )
		{
			*outRef = NULL;
			err = kPGPError_ItemNotFound;
/* Allow to return subkeys */
#if 0
		} else if( OBJISSUBKEY( *outRef ) ) {
			*outRef = (*outRef)->up;
			pgpAssert( OBJISTOPKEY( *outRef ) );
#endif
		}
	}
	
	pgpAssertErrWithPtr( err, *outRef );
	return( err );
}


/* Link a new key/subkey into the keyid-sorted tree */
/* Use random ordering for now, later balance it */
	void
pgpLinkKeyToTree( PGPKeyDBObj *key, PGPKeyDB *kdb )
{
	PGPKeyDBObj *		cur;
	PGPKeyInfo *		curinfo;
	PGPKeyDBObj **		tree;
	PGPKeyID			keyID;
	PGPKeyID			curID;
	PGPInt32			cmp;

	cur = kdb->treeKey;
	if( IsNull( cur ) )
	{
		kdb->treeKey = key;
		return;
	}
	pgpKeyID8( key, NULL, &keyID );
	for ( ; ; ) {
		curinfo = pgpKeyToKeyInfo( cur );
		pgpKeyID8( cur, NULL, &curID );
		cmp = PGPCompareKeyIDs( &curID, &keyID );
		pgpAssert( cmp != 0 );
		tree = (cmp < 0) ? &curinfo->left : &curinfo->right;
		if( IsNull( *tree ) )
		{
			*tree = key;
			return;
		}
		cur = *tree;
	}
}


/*
 * Public-key encrypt a block of data using the specified key or subkey.
 */
	PGPError
pgpKeyEncrypt_internal( PGPKeyDBObj *key, PGPByte const *inbuf,
	PGPSize inbuflen, PGPPublicKeyMessageFormat format,
	PGPByte **outbuf, PGPSize *outbuflen )
{
	PGPContextRef	context;
	PGPRandomContext *rc;
	PGPPubKey *		pub;
	PGPKeyDBRef		db;
	PGPSize			buflen;
	PGPInt32		i;

	db = PGPPeekKeyDBObjKeyDB( key );
	context = PGPPeekKeyDBContext( db );
	rc = pgpContextGetX9_17RandomContext( context );

	pub = pgpKeyPubKey( key, PGP_PKUSE_ENCRYPT );
	if( IsNull( pub ) )
		return kPGPError_KeyUnusableForEncryption;

	buflen = pgpPubKeyMaxesk( pub, format );
	*outbuf = pgpContextMemAlloc( context, buflen, 0 );
	if( IsNull( *outbuf ) )
	{
		pgpPubKeyDestroy( pub );
		return kPGPError_OutOfMemory;
	}

	i = pgpPubKeyEncrypt (pub, inbuf, inbuflen, *outbuf, outbuflen, rc,format);
	pgpPubKeyDestroy( pub );
	if (i < 0)
	{
		PGPFreeData( *outbuf );
		*outbuf = NULL;
		*outbuflen = 0;
		return (PGPError)i;
	}
	return kPGPError_NoErr;
}

	PGPError
pgpKeyEncrypt( PGPKeyDBObj *key, PGPByte const *inbuf,
	PGPSize inbuflen, PGPPublicKeyMessageFormat format,
	PGPByte **outbuf, PGPSize *outbuflen )
{
	if( pgpFrontEndKey( key ) )
	{
		/* Front end, call back end */
		PGPContextRef context = PGPPeekKeyDBObjContext( key );
		return pgpKeyEncrypt_back( context, pgpKeyDBObjID( key ), inbuf,
								   inbuflen, format, outbuf, outbuflen );
	} else {
		/* Back end */
		return pgpKeyEncrypt_internal( key, inbuf, inbuflen, format,
									   outbuf, outbuflen );
	}
}


/*
 * Public-key decrypt a block of data using the specified key or subkey.
 */
	PGPError
pgpKeyDecrypt_internal(PGPKeyDBObj *key, PGPByte const *passphrase,
	PGPSize pplen, PGPBoolean hashedPhrase, PGPUInt32 cacheTimeOut,
	PGPBoolean cacheGlobal, PGPByte const *inbuf, PGPSize inbuflen,
	PGPPublicKeyMessageFormat format, PGPByte **outbuf, PGPSize *outbuflen )
{
	PGPContextRef	context;
	PGPSecKey *		sec;
	PGPKeyDBRef		db;
	PGPSize			buflen;
	PGPInt32		i;
	PGPError		err = kPGPError_NoErr;

	db = PGPPeekKeyDBObjKeyDB( key );
	context = PGPPeekKeyDBContext( db );

	sec = pgpSecSecKey( key, PGP_PKUSE_ENCRYPT );
	if( IsNull( sec ) )
		return kPGPError_KeyUnusableForEncryption;

	/* Try to unlock decryption key with passphrase */
#if 1
	if( IsPGPError( err = pgpSecKeyUnlockWithCache( sec, passphrase, pplen,
													hashedPhrase, cacheTimeOut,
													cacheGlobal ) ) )
	{
		pgpSecKeyDestroy( sec );
		return err;
	}
	if( OBJISSUBKEY( key )  &&  cacheTimeOut > 0 )
	{
		/* Try unlocking top-level key too to refresh cache */
		PGPSecKey *topsec = pgpSecSecKey( key->up, PGP_PKUSE_SIGN );
		if( IsntNull( topsec ) &&
			!pgpMemoryEqual(sec->keyID, topsec->keyID, sizeof(sec->keyID) ) )
		{
			(void)pgpSecKeyUnlockWithCache( topsec, passphrase, pplen,
											hashedPhrase, cacheTimeOut,
											cacheGlobal );
			pgpSecKeyDestroy( topsec );
		}
	}
#else
	if( pgpSecKeyIslocked( sec ) )
	{
		i = pgpSecKeyUnlock (sec, (char *)passphrase, pplen,
							 hashedPhrase );
		if( i <= 0 )
		{
			pgpSecKeyDestroy( sec );
			return kPGPError_BadPassphrase;
		}
	}
#endif

	buflen = pgpSecKeyMaxdecrypted( sec, format );
	*outbuf = pgpContextMemAlloc( context, buflen, 0 );
	if( IsNull( *outbuf ) )
	{
		pgpSecKeyDestroy( sec );
		return kPGPError_OutOfMemory;
	}

	i = pgpSecKeyDecrypt (sec, inbuf, inbuflen, *outbuf, outbuflen,
						  NULL, 0, format);
	pgpSecKeyDestroy( sec );
	if (i < 0)
	{
		PGPFreeData( *outbuf );
		*outbuf = NULL;
		*outbuflen = 0;
		return (PGPError)i;
	}
	return kPGPError_NoErr;
}


	PGPError
pgpKeyDecrypt( PGPKeyDBObj *key, PGPByte const *passphrase, PGPSize pplen,
	PGPBoolean hashedPhrase, PGPUInt32 cacheTimeOut, PGPBoolean cacheGlobal,
	PGPByte const *inbuf, PGPSize inbuflen, PGPPublicKeyMessageFormat format,
	PGPByte **outbuf, PGPSize *outbuflen )
{
	if( pgpFrontEndKey( key ) )
	{
		/* Front end, call back end */
		PGPContextRef context = PGPPeekKeyDBObjContext( key );
		return pgpKeyDecrypt_back( context, pgpKeyDBObjID( key ), passphrase,
								   pplen, hashedPhrase, cacheTimeOut,
								   cacheGlobal, inbuf, inbuflen, format,
								   outbuf, outbuflen );
	} else {
		/* Back end */
		return pgpKeyDecrypt_internal( key, passphrase, pplen, hashedPhrase,
									   cacheTimeOut, cacheGlobal, inbuf,
									   inbuflen, format, outbuf, outbuflen );
	}
}

/*
 * Public-key verify a block of signed data using the specified key or subkey.
 * Return error, or 0 if there were no errors as such but it did not check,
 * or 1 if it checked OK.
 */
	PGPInt32
pgpKeyVerify_internal( PGPKeyDBObj *key, PGPByte const *inbuf,
	PGPSize inbuflen, PGPHashAlgorithm hashalg, PGPByte const *hash,
	PGPSize hashlen, PGPPublicKeyMessageFormat format )
{
	PGPPubKey *			pub;
	PGPInt32			i;
	PGPHashVTBL const *	h = NULL;
	PGPHashVTBL 		hdummy;

	pub = pgpKeyPubKey( key, PGP_PKUSE_SIGN );
	if( IsNull( pub ) )
		return kPGPError_KeyUnusableForSignature;

	if( hashalg != (PGPHashAlgorithm) 0 )
		h = pgpHashByNumber( hashalg );
	if( IsNull( h ) )
	{
		pgpClearMemory( &hdummy, sizeof( hdummy ) );
		hdummy.hashsize = hashlen;
		h = &hdummy;
	}

	i = pgpPubKeyVerify( pub, inbuf, inbuflen, h, hash, format );
	pgpPubKeyDestroy( pub );
	return i;
}

	PGPInt32
pgpKeyVerify( PGPKeyDBObj *key, PGPByte const *inbuf,
	PGPSize inbuflen, PGPHashAlgorithm hashalg, PGPByte const *hash,
	PGPSize hashlen, PGPPublicKeyMessageFormat format )
{
	if( pgpFrontEndKey( key ) )
	{
		/* Front end, call back end */
		PGPContextRef context = PGPPeekKeyDBObjContext( key );
		return pgpKeyVerify_back( context, pgpKeyDBObjID( key ), inbuf,
								  inbuflen, hashalg, hash, hashlen, format );
	} else {
		/* Back end */
		return pgpKeyVerify_internal( key, inbuf, inbuflen, hashalg,
									  hash, hashlen, format );
	}
}

/*
 * Public-key sign a block of data using the specified key or subkey.
 */
	PGPError
pgpKeySign_internal(PGPKeyDBObj *key, PGPByte const *passphrase, PGPSize pplen,
	PGPBoolean hashedPhrase, PGPUInt32 cacheTimeOut, PGPBoolean cacheGlobal,
	PGPHashAlgorithm hashalg, PGPByte const *hash, PGPSize hashlen,
	PGPPublicKeyMessageFormat format, PGPByte **outbuf, PGPSize *outbuflen )
{
	PGPContextRef		context;
	PGPHashVTBL const *	h = NULL;
	PGPHashVTBL 		hdummy;
	PGPRandomContext *	rc;
	PGPSecKey *			sec;
	PGPKeyDBRef			db;
	PGPSize				buflen;
	PGPInt32			i;
	PGPError			err = kPGPError_NoErr;

	db = PGPPeekKeyDBObjKeyDB( key );
	context = PGPPeekKeyDBContext( db );
	rc = pgpContextGetX9_17RandomContext( context );

	sec = pgpSecSecKey( key, PGP_PKUSE_SIGN );
	if( IsNull( sec ) )
		return kPGPError_KeyUnusableForSignature;

	/* Try to unlock signature key with passphrase */
#if 1
	if( IsPGPError( err = pgpSecKeyUnlockWithCache( sec, passphrase, pplen,
													hashedPhrase, cacheTimeOut,
													cacheGlobal ) ) )
	{
		pgpSecKeyDestroy( sec );
		return err;
	}
	if( pgpKeyVersion( key ) > PGPVERSION_3  &&  cacheTimeOut > 0 )
	{
		/* Try unlocking subkey too to refresh cache */
		PGPSecKey *subsec = pgpSecSecKey( key, PGP_PKUSE_ENCRYPT );
		if( IsntNull( subsec ) &&
			!pgpMemoryEqual(sec->keyID, subsec->keyID, sizeof(sec->keyID) ) )
		{
			(void)pgpSecKeyUnlockWithCache( subsec, passphrase, pplen,
											hashedPhrase, cacheTimeOut,
											cacheGlobal );
			pgpSecKeyDestroy( subsec );
		}
	}
#else
	if( pgpSecKeyIslocked( sec ) )
	{
		i = pgpSecKeyUnlock (sec, (char *)passphrase, pplen, hashedPhrase );
		if( i <= 0 )
		{
			pgpSecKeyDestroy( sec );
			return kPGPError_BadPassphrase;
		}
	}
#endif

	buflen = pgpSecKeyMaxsig( sec, format );
	*outbuf = pgpContextMemAlloc( context, buflen, 0 );
	if( IsNull( *outbuf ) )
	{
		pgpSecKeyDestroy( sec );
		return kPGPError_OutOfMemory;
	}

	if( hashalg != (PGPHashAlgorithm) 0 )
		h = pgpHashByNumber( hashalg );
	if( IsNull( h ) )
	{
		pgpClearMemory( &hdummy, sizeof( hdummy ) );
		hdummy.hashsize = hashlen;
		h = &hdummy;
	}

	i = pgpSecKeySign (sec, h, hash, *outbuf, outbuflen, rc, format );
	pgpSecKeyDestroy( sec );
	if (i < 0)
	{
		PGPFreeData( *outbuf );
		*outbuf = NULL;
		*outbuflen = 0;
		return (PGPError)i;
	}
	return kPGPError_NoErr;
}

	PGPError
pgpKeySign( PGPKeyDBObj *key, PGPByte const *passphrase, PGPSize pplen,
	PGPBoolean hashedPhrase, PGPUInt32 cacheTimeOut, PGPBoolean cacheGlobal,
	PGPHashAlgorithm hashalg, PGPByte const *hash, PGPSize hashlen,
	PGPPublicKeyMessageFormat format, PGPByte **outbuf, PGPSize *outbuflen )
{
	if( pgpFrontEndKey( key ) )
	{
		/* Front end, call back end */
		PGPContextRef context = PGPPeekKeyDBObjContext( key );
		return pgpKeySign_back( context, pgpKeyDBObjID( key ), passphrase,
								pplen, hashedPhrase, cacheTimeOut, cacheGlobal,
								hashalg, hash, hashlen, format, outbuf,
								outbuflen );
	} else {
		/* Back end */
		return pgpKeySign_internal( key, passphrase, pplen, hashedPhrase,
									cacheTimeOut, cacheGlobal, hashalg, hash,
									hashlen, format, outbuf, outbuflen );
	}
}

	PGPBoolean
pgpSecPassphraseOK_internal( PGPKeyDBObj *key, PGPByte const *passphrase,
	PGPSize pplen, PGPBoolean hashedPhrase, PGPUInt32 cacheTimeOut,
	PGPBoolean cacheGlobal )
{
	PGPSecKey *		sec;
	PGPError		err = kPGPError_NoErr;

	sec = pgpSecSecKey( key, 0 );
	if( IsNull( sec ) )
		return FALSE;

#if 1
	err = pgpSecKeyUnlockWithCache( sec, passphrase, pplen, hashedPhrase,
									cacheTimeOut, cacheGlobal );
	pgpSecKeyDestroy( sec );
	return IsntPGPError( err );
#else
	if( !pgpSecKeyIslocked( sec ) )
		return TRUE;
	/* Try to unlock key with passphrase */
	i = pgpSecKeyUnlock (sec, (char *)passphrase, pplen, hashedPhrase );
	pgpSecKeyDestroy( sec );
	if( i <= 0 )
		return FALSE;
	return TRUE;
#endif
}

	PGPBoolean
pgpSecPassphraseOK( PGPKeyDBObj *key, PGPByte const *passphrase,
	PGPSize pplen, PGPBoolean hashedPhrase, PGPUInt32 cacheTimeOut,
	PGPBoolean cacheGlobal)
{
	if( pgpFrontEndKey( key ) )
	{
		/* Front end, call back end */
		PGPContextRef context = PGPPeekKeyDBObjContext( key );
		return pgpSecPassphraseOK_back( context, pgpKeyDBObjID( key ),
										passphrase, pplen, hashedPhrase,
										cacheTimeOut, cacheGlobal );
	} else {
		/* Back end */
		return pgpSecPassphraseOK_internal( key, passphrase, pplen,
											hashedPhrase, cacheTimeOut,
											cacheGlobal );
	}
}

	PGPError
pgpKeyMaxSizes_internal( PGPKeyDBObj *key, PGPUInt32 *maxEncryption,
	PGPUInt32 *maxDecryption, PGPUInt32 *maxSignature,
	PGPPublicKeyMessageFormat format )
{
	PGPPubKey *		pub;

	if( IsntNull( maxEncryption ) )
		*maxEncryption = 0;
	if( IsntNull( maxDecryption ) )
		*maxDecryption = 0;
	if( IsntNull( maxSignature ) )
		*maxSignature = 0;

	if( IsntNull( maxEncryption ) || IsntNull( maxDecryption ) )
	{
		pub = pgpKeyPubKey( key, PGP_PKUSE_ENCRYPT );
		if( IsntNull( pub ) )
		{
			if( IsntNull( maxEncryption ) )
				*maxEncryption = pgpPubKeyMaxesk( pub, format );
			if( IsntNull( maxDecryption ) )
				*maxDecryption = pgpPubKeyMaxdecrypted( pub, format );
		}
		pgpPubKeyDestroy( pub );
	}

	if( IsntNull( maxSignature ) )
	{
		pub = pgpKeyPubKey( key, PGP_PKUSE_SIGN );
		if( IsntNull( pub ) )
		{
			if( IsntNull( maxSignature ) )
				*maxSignature = pgpPubKeyMaxsig( pub, format );
		}
		pgpPubKeyDestroy( pub );
	}

	return kPGPError_NoErr;
}

	PGPError
pgpKeyMaxSizes( PGPKeyDBObj *key, PGPUInt32 *maxEncryption,
	PGPUInt32 *maxDecryption, PGPUInt32 *maxSignature,
	PGPPublicKeyMessageFormat format )
{
	if( pgpFrontEndKey( key ) )
	{
		/* Front end, call back end */
		PGPContextRef context = PGPPeekKeyDBObjContext( key );
		return pgpKeyMaxSizes_back( context, pgpKeyDBObjID( key ),
									maxEncryption, maxDecryption, maxSignature,
									format );
	} else {
		/* Back end */
		return pgpKeyMaxSizes_internal( key, maxEncryption, maxDecryption,
										maxSignature, format );
	}
}

	PGPError
pgpSecProperties_internal( PGPKeyDBObj *key, PGPBoolean *needsPassphrase,
	PGPBoolean *isSecretShared, PGPCipherAlgorithm *lockAlg,
	PGPUInt32 *lockBits )
{
	PGPSecKey *			sec;
	PGPContextRef		context;
	PGPEnv const *		pgpEnv;
	PGPSize				lockbytes;
	PGPStringToKeyType	s2ktype;
	PGPError			err;

	*needsPassphrase = FALSE;
	*isSecretShared = FALSE;
	*lockAlg = kPGPCipherAlgorithm_None;
	*lockBits = 0;

	context = PGPPeekKeyDBObjContext( key );
	pgpEnv = pgpContextGetEnvironment( context );

	sec = pgpSecSecKey( key, 0 );
	if( IsNull( sec ) )
		return kPGPError_InvalidProperty;

	*needsPassphrase = pgpSecKeyIslocked( sec );
	if( !*needsPassphrase ) {
		/* Use key's preferred alg if key is not locked */
		PGPByte const *prefAlgs;
		PGPSize prefAlgsLength;
		PGPBoolean hashed;
		PGPUInt32 i;
		prefAlgs = pgpKeyFindSubpacket ( PGPPeekKeyDBObjKey( key ),
				SIGSUB_PREFERRED_ENCRYPTION_ALGS, 0,
				&prefAlgsLength, NULL, &hashed, NULL, NULL, &err);
		for( i = 0; hashed && (i < prefAlgsLength); ++i ) {
			PGPCipherVTBL const *cipher = pgpCipherGetVTBL( (PGPCipherAlgorithm) prefAlgs[i] );
			if( IsntNull( cipher ) ) {
				*lockAlg = cipher->algorithm;
				*lockBits = (PGPUInt32) cipher->keysize * 8;
				break;
			}
		}

		if( *lockAlg == kPGPCipherAlgorithm_None ) {
			/* Use defaults if we don't know any preferred algs */
			PGPCipherVTBL const *cipher = pgpCipherDefaultKey( pgpEnv );
			*lockAlg = cipher->algorithm;
			*lockBits = (PGPUInt32) cipher->keysize * 8;
		}
	} else {
		err = pgpSecKeyLockingalgorithm( sec, lockAlg, &lockbytes );
		if( IsPGPError( err ) )
			return err;
		*lockBits = (PGPUInt32) lockbytes * 8;
		pgpSecKeyS2Ktype( sec, &s2ktype );
		if( s2ktype == kPGPStringToKey_LiteralShared )
			*isSecretShared = TRUE;
	}

	pgpSecKeyDestroy( sec );
	return kPGPError_NoErr;
}


	PGPError
pgpSecProperties( PGPKeyDBObj *key, PGPBoolean *needsPassphrase,
	PGPBoolean *isSecretShared, PGPCipherAlgorithm *lockAlg,
	PGPUInt32 *lockBits )
{
	if( pgpFrontEndKey( key ) )
	{
		/* Front end, call back end */
		PGPContextRef context = PGPPeekKeyDBObjContext( key );
		return pgpSecProperties_back( context, pgpKeyDBObjID( key ),
									  needsPassphrase, isSecretShared, lockAlg,
									  lockBits );
	} else {
		return pgpSecProperties_internal( key, needsPassphrase, isSecretShared,
										  lockAlg, lockBits );
	}
}


/* Clear trust flags from specified object */
	static void
sKeyDBObjClearTrust( PGPKeyDBObj *obj )
{
	PGPKeyInfo *kinfo;
	PGPUserIDInfo *uinfo;
	PGPSigInfo *sinfo;
	PGPCRLInfo *cinfo;
	PGPUnkInfo *xinfo;
	
	switch( pgpObjectType( obj ) )
	{
	case RINGTYPE_KEY:
	case RINGTYPE_SUBKEY:
		kinfo = pgpKeyToKeyInfo( obj );
		kinfo->signedTrust = 0;
		kinfo->trust = 0;
		break;
	case RINGTYPE_USERID:
		uinfo = pgpUserIDToUserIDInfo( obj );
		uinfo->validity = 0;
		uinfo->oldvalidity = 0;
		uinfo->confidence = 0;
		uinfo->valid = 0;
		break;
	case RINGTYPE_SIG:
		sinfo = pgpSigToSigInfo( obj );
		sinfo->trust = 0;
		break;
	case RINGTYPE_CRL:
		cinfo = pgpCRLToCRLInfo( obj );
		cinfo->trust = 0;
		break;
	case RINGTYPE_UNK:
		xinfo = pgpUnkToUnkInfo( obj );
		xinfo->trust = 0;
		break;
	default:
		pgpAssert( 0 );
		break;
	}
}


/* Mark object and all its children as removed */
	PGPError
pgpMarkKeyDBObjectDeleted( PGPKeyDBObj *obj )
{
	PGPKeyDBObj *		child;
	PGPKeyDBObj *		gchild;
	PGPBoolean			front;

	front = pgpFrontEndKey( obj );
	obj->objflags |= kPGPKeyDBObjFlags_Deleted;
	obj->userVal = 0;
	if( front )
		pgpKeyDBObjRefresh( obj, FALSE );
	else
		sKeyDBObjClearTrust( obj );
	for( child = obj->down; IsntNull(child); child = child->next )
	{
		if( !pgpKeyDBObjIsReal( child ) )
			continue;
		child->objflags |= kPGPKeyDBObjFlags_Deleted;
		child->userVal = 0;
		if( front )
			pgpKeyDBObjRefresh( child, FALSE );
		else
			sKeyDBObjClearTrust( child );
		for( gchild = child->down; IsntNull(gchild); gchild = gchild->next )
		{
			if( !pgpKeyDBObjIsReal( gchild ) )
				continue;
			gchild->objflags |= kPGPKeyDBObjFlags_Deleted;
			gchild->userVal = 0;
			if( front )
				pgpKeyDBObjRefresh( gchild, FALSE );
			else
				sKeyDBObjClearTrust( gchild );
		}
	}
	return kPGPError_NoErr;
}


	static void
sSetInfo( PGPKeyDBObj *obj, PGPByte *infobuf, PGPSize infosize )
{
	PGPKeyDB *kdb;
	PGPByte *info;
	PGPUInt32 id;

	kdb = PGPPeekKeyDBObjKeyDB( obj );
	info = (PGPByte *)memPoolAlloc( &kdb->objPool, infosize, sizeof (void *) );
	pgpCopyMemory( infobuf, info, infosize );
	PGPFreeData( infobuf );
	id = obj->idinfo.id;
	obj->idinfo.info = (PGPKeyDBObjInfo *)info;
	obj->objflags &= ~kPGPKeyDBObjFlags_ID;
	*sObjectIDPointer( obj ) = id;
}


/* Refresh data associated with a key which has changed */
	PGPError
pgpKeyDBObjRefresh( PGPKeyDBObj *obj, PGPBoolean recurse )
{
	PGPKeyDBObj *	child;

	pgpAssert( pgpFrontEndKey( obj ) );
	if( !(obj->objflags & kPGPKeyDBObjFlags_ID) )
	{
		PGPUInt32 id;
		id = pgpKeyDBObjID( obj );
		obj->idinfo.id = id;
		obj->objflags |= kPGPKeyDBObjFlags_ID;
		/* Free old info and data here if we were doing it that way */
	}
	if( recurse && !OBJISSIG( obj ) )
	{
		for( child = obj->down; IsntNull( child ); child = child->next )
		{
			pgpKeyDBObjRefresh( child, TRUE );
		}
	}
	return kPGPError_NoErr;
}

/* Refresh the keys in the list, and free it */
	PGPError
pgpKeyRefreshFromList( PGPKeyDB *kdb, PGPUInt32 *list, PGPSize listsize )
{
	PGPUInt32		i;
	PGPKeyDBObj *	obj;

	listsize /= sizeof( PGPUInt32 );
	for( i=0; i<listsize; ++i )
	{
		if( IsntPGPError( PGPFindNode( kdb->idToObj, list[i],
									   (PGPUserValue *)&obj ) ) )
			pgpKeyDBObjRefresh( obj, TRUE );
	}
	if( IsntNull( list ) )
		PGPFreeData( list );
	return kPGPError_NoErr;
}
			


	PGPKeyInfo *
pgpKeyToKeyInfo( PGPKeyDBObj const *obj )
{
	PGPByte *infobuf;
	PGPSize infosize;
	PGPKeyDB *kdb;
	PGPKeyInfo *kinfo;
	PGPUInt32 id;
	PGPUserValue idobj;

	if( obj->objflags & kPGPKeyDBObjFlags_ID )
	{
		PGPContextRef context = PGPPeekKeyDBObjContext( (PGPKeyDBObj *)obj );
		pgpFetchKeyInfo_back( context, obj->idinfo.id, &infobuf, &infosize);
		pgpAssert( infosize == sizeof (PGPKeyInfo) );
		sSetInfo( (PGPKeyDBObj *)obj, infobuf, infosize );
		/* Update pointers */
		kinfo = &obj->idinfo.info->keyinfo;
		kinfo->data = NULL;
		kdb = PGPPeekKeyDBObjKeyDB( (PGPKeyDBObjRef) obj );
		id = (PGPUInt32) kinfo->sigsby;
		if( id == 0 ||
			IsPGPError( PGPFindNode( kdb->idToObj, id, &idobj ) ) )
			kinfo->sigsby = NULL;
		else
			kinfo->sigsby = (PGPKeyDBObj *)idobj;
	}
	return &obj->idinfo.info->keyinfo;
}

	PGPUserIDInfo *
pgpUserIDToUserIDInfo( PGPKeyDBObj const *obj )
{
	PGPByte *infobuf;
	PGPSize infosize;
	PGPUserIDInfo *uinfo;

	if( obj->objflags & kPGPKeyDBObjFlags_ID )
	{
		PGPContextRef context = PGPPeekKeyDBObjContext( (PGPKeyDBObj *)obj );
		pgpFetchKeyInfo_back( context, obj->idinfo.id, &infobuf, &infosize);
		pgpAssert( infosize == sizeof (PGPUserIDInfo) );
		sSetInfo( (PGPKeyDBObj *)obj, infobuf, infosize );
		uinfo = &obj->idinfo.info->useridinfo;
		uinfo->data = NULL;
	}
	return &obj->idinfo.info->useridinfo;
}

	PGPSigInfo *
pgpSigToSigInfo( PGPKeyDBObj const *obj )
{
	PGPByte *infobuf;
	PGPSize infosize;
	PGPKeyDB *kdb;
	PGPSigInfo *sinfo;
	PGPUInt32 id;
	PGPUserValue idobj;

	if( obj->objflags & kPGPKeyDBObjFlags_ID )
	{
		PGPContextRef context = PGPPeekKeyDBObjContext( (PGPKeyDBObj *)obj );
		pgpFetchKeyInfo_back( context, obj->idinfo.id, &infobuf, &infosize);
		pgpAssert( infosize == sizeof (PGPSigInfo) );
		sSetInfo( (PGPKeyDBObj *)obj, infobuf, infosize );
		/* Update info fields */
		sinfo = &obj->idinfo.info->siginfo;
		sinfo->data = NULL;
		kdb = PGPPeekKeyDBObjKeyDB(  (PGPKeyDBObjRef) obj );
		id = (PGPUInt32) sinfo->by;
		if( id == 0 ||
			IsPGPError( PGPFindNode( kdb->idToObj, id, &idobj ) ) )
			sinfo->by = NULL;
		else
			sinfo->by = (PGPKeyDBObj *)idobj;
		id = (PGPUInt32) sinfo->nextby;
		if( id == 0 ||
			IsPGPError( PGPFindNode( kdb->idToObj, id, &idobj ) ) )
			sinfo->nextby = NULL;
		else
			sinfo->nextby = (PGPKeyDBObj *)idobj;
	}
	return &obj->idinfo.info->siginfo;
}

	PGPCRLInfo *
pgpCRLToCRLInfo( PGPKeyDBObj const *obj )
{
	PGPByte *infobuf;
	PGPSize infosize;
	PGPCRLInfo *cinfo;

	if( obj->objflags & kPGPKeyDBObjFlags_ID )
	{
		PGPContextRef context = PGPPeekKeyDBObjContext( (PGPKeyDBObj *)obj );
		pgpFetchKeyInfo_back( context, obj->idinfo.id, &infobuf, &infosize);
		pgpAssert( infosize == sizeof (PGPCRLInfo) );
		sSetInfo( (PGPKeyDBObj *)obj, infobuf, infosize );
		cinfo = &obj->idinfo.info->crlinfo;
		cinfo->data = NULL;
	}
	return &obj->idinfo.info->crlinfo;
}

	PGPUnkInfo *
pgpUnkToUnkInfo( PGPKeyDBObj const *obj )
{
	PGPByte *infobuf;
	PGPSize infosize;
	PGPUnkInfo *uinfo;

	if( obj->objflags & kPGPKeyDBObjFlags_ID )
	{
		PGPContextRef context = PGPPeekKeyDBObjContext( (PGPKeyDBObj *)obj );
		pgpFetchKeyInfo_back( context, obj->idinfo.id, &infobuf, &infosize);
		pgpAssert( infosize == sizeof (PGPUnkInfo) );
		sSetInfo( (PGPKeyDBObj *)obj, infobuf, infosize );
		uinfo = &obj->idinfo.info->unkinfo;
		uinfo->data = NULL;
	}
	return &obj->idinfo.info->unkinfo;
}




/* True if object is considered deleted */
	PGPBoolean
pgpKeyDBObjIsDeleted( PGPKeyDBObj const *obj )
{
	if( IsNull( obj ) )
		return FALSE;
	return (obj->objflags & kPGPKeyDBObjFlags_Deleted) != 0;
}

/* True if object is a dummy object (subkey or nonlocal signer) */
	PGPBoolean
pgpKeyDBObjIsDummy( PGPKeyDBObj const *obj )
{
	if( IsNull( obj ) )
		return FALSE;
	return (obj->objflags & kPGPKeyDBObjFlags_Dummy) != 0;
}

/* True if object is an X.509 dummy key; these have keyids which are the
 * hash of the X.509 name */
	PGPBoolean
pgpKeyDBObjIsX509Dummy( PGPKeyDBObj const *obj )
{
	if( IsNull( obj ) )
		return FALSE;
	return (obj->objflags & kPGPKeyDBObjFlags_X509) != 0;
}

/* True if object is neither deleted nor dummy */
	PGPBoolean
pgpKeyDBObjIsReal( PGPKeyDBObj const *obj )
{
	if( IsNull( obj ) )
		return FALSE;
	return (obj->objflags & (kPGPKeyDBObjFlags_Deleted
						  | kPGPKeyDBObjFlags_Dummy) ) == 0;
}


	PGPBoolean
pgpKeyDBObjIsValid( PGPKeyDBObj const *		obj)
{
	PGPBoolean	isValid;
	
	isValid	= IsntNull( obj ) && (obj->magic == kPGPKeyDBObjMagic);
			
	return( isValid );
}


	PGPBoolean
pgpKeyIsValid( PGPKeyDBObj const *		key)
{
	PGPBoolean	isValid;
	
	isValid	= pgpKeyDBObjIsValid(key) && OBJISTOPKEY(key);
			
	return( isValid );
}

	PGPBoolean
pgpSubKeyIsValid(
	PGPKeyDBObj const *	subKey)
{
	PGPBoolean	isValid;
	
	isValid	= pgpKeyDBObjIsValid(subKey) && OBJISSUBKEY(subKey);
			
	return( isValid );
}

	PGPBoolean
pgpUserIDIsValid(
	PGPKeyDBObj const *	userID)
{
	PGPBoolean	isValid;
	
	isValid	= pgpKeyDBObjIsValid(userID) && OBJISUSERID(userID);
			
	return( isValid );
}

	PGPBoolean
pgpSigIsValid(
	PGPKeyDBObj const *		cert)
{
	PGPBoolean	isValid;
	
	isValid	= pgpKeyDBObjIsValid(cert) && OBJISSIG(cert);

	return( isValid );
}


#if PGP_DEBUG	/* [ */
	PGPBoolean
pgpaInternalPGPKeyDBObjValid(
	pgpaCallPrefixDef,
	PGPKeyDBObj const *		obj,
	char const *		varName)
{
	pgpaAddrValid(obj, PGPKeyDBObj);
	pgpaFmtMsg((pgpaFmtPrefix,
			"pgpaPGPKeyDBObjValid failed on %s (%p)", varName, obj));

	return pgpaFailed;
}

	PGPBoolean
pgpaInternalPGPKeyValid(
	pgpaCallPrefixDef,
	PGPKeyDBObj const *		key,
	char const *		varName)
{
	pgpaAddrValid(key, PGPKeyDBObj);
	pgpaFmtMsg((pgpaFmtPrefix,
			"pgpaPGPKeyValid failed on %s (%p)", varName, key));

	return pgpaFailed;
}

	PGPBoolean
pgpaInternalPGPSubKeyValid(
	pgpaCallPrefixDef,
	PGPKeyDBObj const *	subKey,
	char const *		varName)
{
	pgpaAddrValid(subKey, PGPKeyDBObj);
/*	pgpaFailIf(subKey->refCount <= 0, (pgpaFmtPrefix, "refCount <= 0"));	*/
	pgpaFmtMsg((pgpaFmtPrefix,
			"pgpaPGPSubKeyValid failed on %s (%p)", varName, subKey));

	return pgpaFailed;
}

	PGPBoolean
pgpaInternalPGPUserIDValid(
	pgpaCallPrefixDef,
	PGPKeyDBObj const *	userID,
	char const *		varName)
{
	pgpaAddrValid(userID, PGPKeyDBObj);
/*	pgpaFailIf(userID->refCount <= 0, (pgpaFmtPrefix, "refCount <= 0"));	*/
	pgpaFmtMsg((pgpaFmtPrefix,
			"pgpaPGPUserIDValid failed on %s (%p)", varName, userID));

	return pgpaFailed;
}

	PGPBoolean
pgpaInternalPGPCertValid(
	pgpaCallPrefixDef,
	PGPKeyDBObj const *		cert,
	char const *		varName)
{
	pgpaAddrValid(cert, PGPKeyDBObj);
/*	pgpaFailIf(cert->refCount <= 0, (pgpaFmtPrefix, "refCount <= 0")); 	*/
	pgpaFmtMsg((pgpaFmtPrefix,
			"pgpaPGPCertValid failed on %s (%p)", varName, cert));

	return pgpaFailed;
}
#endif /* ] PGP_DEBUG */


/*
 * Local Variables:
 * tab-width: 4
 * End:
 * vi: ts=4 sw=4
 * vim: si
 */
